(*
Author:  Christian Sternagel <c.sternagel@gmail.com>
Author:  René Thiemann <rene.thiemann@uibk.ac.at>
License: LGPL (see file COPYING.LESSER)
*)
chapter \<open>Term Rewriting\<close>

theory Term_Rewriting
  imports
    First_Order_Terms.Subterm_and_Context
    Relation_Closure
begin


text \<open>
  A rewrite rule is a pair of terms. A term rewrite system (TRS) is a set of rewrite rules.
\<close>
type_synonym ('f, 'v) rule = "('f, 'v) term \<times> ('f, 'v) term"
type_synonym ('f, 'v) trs  = "('f, 'v) rule set"

inductive_set rstep :: "_ \<Rightarrow> ('f, 'v) term rel" for R :: "('f, 'v) trs"
  where
    rstep: "\<And>C \<sigma> l r. (l, r) \<in> R \<Longrightarrow> s = C\<langle>l \<cdot> \<sigma>\<rangle> \<Longrightarrow> t = C\<langle>r \<cdot> \<sigma>\<rangle> \<Longrightarrow> (s, t) \<in> rstep R"

lemma rstep_induct_rule [case_names IH, induct set: rstep]:
  assumes "(s, t) \<in> rstep R"
    and "\<And>C \<sigma> l r. (l, r) \<in> R \<Longrightarrow> P (C\<langle>l \<cdot> \<sigma>\<rangle>) (C\<langle>r \<cdot> \<sigma>\<rangle>)"
  shows "P s t"
  using assms by (induct) simp

text \<open>
  An alternative induction scheme that treats the rule-case, the
  substition-case, and the context-case separately.
\<close>
lemma rstep_induct [consumes 1, case_names rule subst ctxt]:
  assumes "(s, t) \<in> rstep R"
    and rule: "\<And>l r. (l, r) \<in> R \<Longrightarrow> P l r"
    and subst: "\<And>s t \<sigma>. P s t \<Longrightarrow> P (s \<cdot> \<sigma>) (t \<cdot> \<sigma>)"
    and ctxt: "\<And>s t C. P s t \<Longrightarrow> P (C\<langle>s\<rangle>) (C\<langle>t\<rangle>)"
  shows "P s t"
  using assms by (induct) auto


lemmas rstepI = rstep.intros [intro]

lemmas rstepE = rstep.cases [elim]

lemma rstep_ctxt [intro]: "(s, t) \<in> rstep R \<Longrightarrow> (C\<langle>s\<rangle>, C\<langle>t\<rangle>) \<in> rstep R"
  by (force simp flip: ctxt_ctxt_compose)

lemma rstep_rule [intro]: "(l, r) \<in> R \<Longrightarrow> (l, r) \<in> rstep R"
  using rstep.rstep [where C = \<box> and \<sigma> = Var and R = R] by simp

lemma rstep_subst [intro]: "(s, t) \<in> rstep R \<Longrightarrow> (s \<cdot> \<sigma>, t \<cdot> \<sigma>) \<in> rstep R"
  by (force simp flip: subst_subst_compose)

lemma rstep_empty [simp]: "rstep {} = {}"
  by auto

lemma rstep_mono: "R \<subseteq> S \<Longrightarrow> rstep R \<subseteq> rstep S"
  by force

lemma rstep_union: "rstep (R \<union> S) = rstep R \<union> rstep S"
  by auto

lemma rstep_converse [simp]: "rstep (R\<inverse>) = (rstep R)\<inverse>"
  by auto

interpretation subst: rel_closure "\<lambda>\<sigma> t. t \<cdot> \<sigma>" Var "\<lambda>x y. y \<circ>\<^sub>s x" by (standard) auto
declare subst.closure.induct [consumes 1, case_names subst, induct pred: subst.closure]
declare subst.closure.cases [consumes 1, case_names subst, cases pred: subst.closure]

interpretation ctxt: rel_closure "ctxt_apply_term" \<box> "(\<circ>\<^sub>c)" by (standard) auto
declare ctxt.closure.induct [consumes 1, case_names ctxt, induct pred: ctxt.closure]
declare ctxt.closure.cases [consumes 1, case_names ctxt, cases pred: ctxt.closure]

lemma rstep_eq_closure: "rstep R = ctxt.closure (subst.closure R)"
  by (force elim: ctxt.closure.cases subst.closure.cases)

lemma ctxt_closed_rstep [intro]: "ctxt.closed (rstep R)"
  by (simp add: rstep_eq_closure ctxt.closed_closure)

lemma ctxt_closed_one:
  "ctxt.closed r \<Longrightarrow> (s, t) \<in> r \<Longrightarrow> (Fun f (ss @ s # ts), Fun f (ss @ t # ts)) \<in> r"
  using ctxt.closedD [of r s t "More f ss \<box> ts"] by auto

lemma args_steps_imp_steps:
  assumes "ctxt.closed r"
    and len: "length ts = length us"
    and "\<forall>i < length us. (ts ! i, us ! i) \<in> r\<^sup>*"
  shows "(Fun f ts, Fun f us) \<in> r\<^sup>*"
proof -
  have "(Fun f ts, Fun f (take i us @ drop i ts)) \<in> r\<^sup>*" if "i \<le> length us" for i
    using that
  proof (induct i)
    case (Suc i)
    then have "(Fun f ts, Fun f (take i us @ drop i ts)) \<in> r\<^sup>*" by simp
    moreover have "take i us @ drop i ts = take i us @ ts ! length (take i us) # drop (Suc i) ts"
      using Suc and len by (auto simp: Cons_nth_drop_Suc min_def)
    moreover have "take (Suc i) us @ drop (Suc i) ts = take i us @ us ! length (take i us) # drop (Suc i) ts"
      using Suc and len by (auto simp: min_def take_Suc_conv_app_nth)
    moreover have "(Fun f (take i us @ ts ! length (take i us) # drop (Suc i) ts),
    Fun f (take i us @ us ! length (take i us) # drop (Suc i) ts)) \<in> r\<^sup>*"
      using assms and Suc by (intro ctxt_closed_one) auto
    ultimately show ?case by simp
  qed simp
  note this [of "length us"] and len
  then show ?thesis by simp
qed

end
