theory Matrix_TRS_Impl
  imports 
    Matrix_TRS
    Certification_Monads.Check_Monad
    Certification_Monads.Error_Monad
    Show.Shows_Literal
    Jordan_Normal_Form.Matrix_Impl
    Matrix_Core_Order_Impl
begin

lemma squared_int_mat: "squared_ring_mat_with_assms (\<lambda> x. x \<ge> 1) (1 :: int) (\<lambda> x y. x > y)" 
  by (unfold_locales, auto)

lemma squared_ring_mat_delta: "0 < \<delta> \<Longrightarrow> \<delta> \<le> 1 \<Longrightarrow> squared_ring_mat_with_assms delta_mono 1 (delta_gt \<delta>)" 
  using delta_interpretation[of \<delta> 1]
  unfolding squared_ring_mat_with_assms_def squared_ring_mat_with_assms_axioms_def
  by (auto simp: delta_gt_def)


abbreviation "I \<equiv> LPI" 
abbreviation "Ip \<equiv> LPIp" 


subsection \<open>Basic checks related to Matrix_Base\<close>
\<comment> \<open>checks on: carrier_mat n n, positive cone, strictly positive cone,
    E_I, M_I, P_I, set of indices I\<close>


definition check_carrier :: "nat \<Rightarrow> 'a mat \<Rightarrow> showsl check" where
"check_carrier n M = do {
  check (dim_row M = n) (showsl_lit (STR ''Expected '') o showsl n o
         showsl_lit (STR '' rows in matrix, got '') o showsl (dim_row M));
  check (dim_col M = n) (showsl_lit (STR ''Expected '') o showsl n o
         showsl_lit (STR '' columns in matrix, got '') o showsl (dim_col M))
}"
lemma check_carrier[simp]: assumes "isOK (check_carrier n M)"
  shows "M \<in> carrier_mat n n"
  using assms[unfolded check_carrier_def, simplified] by auto
  

definition check_N :: "nat \<Rightarrow> _ mat \<Rightarrow> showsl check" where
"check_N n M = (do {
    check_carrier n M;
    check (\<forall>i < n. \<forall>j < n. M $$ (i,j) \<ge> 0)
         (showsl_lit (STR ''Expected all matrix element to be non-negative: '')
          o showsl (mat_to_list M))
})"
lemma check_N[simp]: assumes "isOK (check_N n M)"
  shows "M \<in> squared_ring_mat.N n"
  using assms[unfolded check_N_def, simplified] 
  by (auto simp: squared_ring_mat.N_compo_greater_eq_zero2)

lemma check_N_carr[simp]: assumes "isOK (check_N n M)"
  shows "isOK (check_carrier n M)"
  using assms[unfolded check_N_def, simplified] by auto

definition check_indices :: "nat \<Rightarrow> nat list \<Rightarrow> showsl check" where
"check_indices n ids = (do {
    check (set ids \<noteq> {}) (showsl_lit (STR ''Set of indices should not be empty''));
    check (set ids \<subseteq> {0..< n}) (showsl_lit (STR ''I should be a subset of {0..n-1}''))
})"
lemma check_indices[simp]: assumes "isOK (check_indices n ids)"
  shows "set ids \<noteq> {} \<and> set ids \<subseteq> {0..< n}"
  using assms[unfolded check_indices_def, simplified] by auto


definition check_E\<^sub>I :: "nat \<Rightarrow> nat list \<Rightarrow> _ mat \<Rightarrow> showsl check" where
"check_E\<^sub>I n ids M = (do {
    check (\<forall>i \<in> set ids. M $$ (i,i) \<ge> 1)
         (showsl_lit (STR ''[E_I set] Expected all diagonal elements to be at least 1: '')
          o showsl (mat_to_list M))
})"

lemma check_E\<^sub>I[simp]: assumes "isOK (check_E\<^sub>I n ids M)"
  and "M \<in> squared_ring_mat.N n" 
  shows "M \<in> squared_ring_mat.E\<^sub>I n (set ids)"
  using assms[unfolded check_E\<^sub>I_def, simplified] 
  by (auto simp: squared_ring_mat.E\<^sub>I_def) 



definition check_P\<^sub>I :: "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> _ mat \<Rightarrow> showsl check" where
"check_P\<^sub>I n gt ids M = (do {
    check (\<exists>i \<in> set ids. \<exists>j \<in> set ids. gt (M $$ (i,j)) 0)
         (showsl_lit (STR ''[P_I set] Expected at least one positive element''))
})"
lemma check_P\<^sub>I[simp]: assumes "isOK (check_P\<^sub>I n gt ids M)"
  and "M \<in> squared_ring_mat.N n" 
  shows "M \<in> squared_ring_mat.P\<^sub>I n gt (set ids)"
  using assms[unfolded check_P\<^sub>I_def, simplified] 
  by (auto simp: squared_ring_mat.P\<^sub>I_def)


definition check_M\<^sub>I :: "nat \<Rightarrow> nat list \<Rightarrow> _ mat \<Rightarrow> showsl check" where
"check_M\<^sub>I n ids M = (do {
    check (\<forall>i \<in> set ids. \<exists>j \<in> set ids. M $$ (i,j) \<ge> 1)
         (showsl_lit (STR ''[M_I set] Expected at least one element element per row that is at least 1 '')
          o showsl (mat_to_list M))
})"

lemma check_M\<^sub>I[simp]: assumes "isOK (check_M\<^sub>I n ids M)"
  and "M \<in> squared_ring_mat.N n" 
shows "M \<in> squared_ring_mat.M\<^sub>I n (set ids)"
  using assms[unfolded check_M\<^sub>I_def, simplified] 
  by (auto simp: squared_ring_mat.M\<^sub>I_def)

definition check_L\<^sub>I :: "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> _ mat list \<Rightarrow> showsl check" where
"check_L\<^sub>I n gt ids Ms = (do {
    check (\<forall>i \<in> set ids. \<exists> M \<in> set Ms. \<exists>j \<in> set ids. gt (M $$ (i,j)) 0)
         (showsl_lit (STR ''[L_I set] Expected at least one positive element per row for coefficients: '')
          o showsl_list (map mat_to_list Ms))
})"

subsection \<open>Terms-related checks, related to Matrix_TRS and Linear_Poly_Interpretation\<close>
\<comment> \<open>checks on: Variable interpretations being in the strict carrier (see strictly_ordered_semiring),
    having the right number of function interpretation coefficients and them begin in the strict carrier\<close>


type_synonym 'v variables = "'v list" \<comment> \<open>the list of variables occurring in a TRS\<close>
type_synonym ('v,'a) assignment = "('v, 'a mat) p_ass"
  \<comment> \<open>the list of variables occurring in a TRS and their corresponding interpretation\<close>

definition check_coeffs_length :: "('f :: showl,_) fun_coeffs \<Rightarrow> showsl check"
  where
"check_coeffs_length fc = check_allm (\<lambda>fx. case fx of ((f,n), (cs,c)) \<Rightarrow>
 check (length cs = n) (showsl_lit (STR ''Expected as many multiplicative coefficients as the arity of '') o showsl f o
       showsl_lit (STR ''. Got '') o showsl (length cs) o showsl_lit (STR '' instead of '') o showsl n)) fc
<+? (\<lambda>str. showsl_lit (STR ''The number of interpretation coefficients (multiplicative, i.e. not the constant coefficient)
 should always match the arity of the corresponding function symbol.\<newline>'') o str)"

lemma check_coeffs_length[simp]: assumes "isOK (check_coeffs_length fc)"
  shows "((f,n),(cs,c)) \<in> set fc \<Longrightarrow> length cs = n"
  using assms[unfolded check_coeffs_length_def, simplified] by auto


definition check_coefficients_N ::
 "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> showsl check"
  where
"check_coefficients_N n ids fc = (do {
    check_allm (\<lambda>fx. case fx of ((f,m),(cs,c)) \<Rightarrow>
       check_allm (check_N n) (c#cs) <+? (\<lambda>str. showsl_lit (STR ''Expected all interpretation coefficients of symbol '')
 o showsl f o showsl_lit (STR '' to be in N\<newline>'') o str)
    ) fc <+? (\<lambda>str. showsl_lit (STR ''Expected all interpretation coefficients of all function symbols used to be in N.\<newline>'') o str)
})"

lemma check_coefficients_N[simp]: assumes "isOK (check_coefficients_N n ids fc)"
  shows "((f,m),(cs,c)) \<in> set fc \<Longrightarrow> set (c#cs) \<subseteq> squared_ring_mat.N n"
  using assms[unfolded check_coefficients_N_def, simplified] by auto

definition check_coefficients_E\<^sub>I ::
 "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> showsl check"
  where
"check_coefficients_E\<^sub>I n ids fc = (do {
    check_coefficients_N n ids fc;
    check_allm (\<lambda>fx. case fx of ((f,m),(cs,c)) \<Rightarrow> (do{
        check_allm (check_E\<^sub>I n ids) cs <+? (\<lambda>str. showsl_lit(STR ''Expected all interpretation multiplicative coefficients of symbol '')
 o showsl f o showsl_lit(STR '' to be in E_I.\<newline>'') o str);
        check (\<exists>ci \<in> set(c#cs). isOK (check_E\<^sub>I n ids ci)) (
          (showsl_lit(STR ''Expected the constant coefficient '') o showsl (mat_to_list c) o
 showsl_lit(STR '' of symbol '') o showsl f o
 showsl_lit(STR '' to be in E_I whenever there are no multiplicative coefficient (i.e. symbol of arity 0).''))
        )
      })
    ) fc <+? (\<lambda>str. showsl_lit(STR ''Expected all interpretation multiplicative coeffs to be in E_I
 and the constant coeff to be in E_I whenever there are none multiplicative coeffs, for all function symbols.\<newline>'') o str)
})"

lemma check_coefficients_E\<^sub>I[simp]: assumes "isOK (check_coefficients_E\<^sub>I n ids fc)"
  shows "((f,m),(cs,c)) \<in> set fc \<Longrightarrow> set cs \<subseteq> squared_ring_mat.E\<^sub>I n (set ids) \<and>
 set (c#cs) \<inter> squared_ring_mat.E\<^sub>I n (set ids) \<noteq> {}"
proof -
  assume as: "((f,m),(cs,c)) \<in> set fc"
  note check = assms[unfolded check_coefficients_E\<^sub>I_def, simplified]
  from check have "isOK (check_coefficients_N n ids fc)" by auto
  
  from check_coefficients_N[OF this as] 
  have inN: "ci \<in> set (c # cs) \<Longrightarrow> ci \<in> squared_ring_mat.N n" for ci by auto
  have "\<forall>ci \<in> set cs. ci \<in> squared_ring_mat.E\<^sub>I n (set ids)"
            "\<exists>ci \<in> set (c#cs). ci \<in> squared_ring_mat.E\<^sub>I n (set ids)"
    by (insert as check check_E\<^sub>I[OF _  inN], fastforce)+
  then show ?thesis by auto
qed
  
  
lemma check_coefficients_E\<^sub>I_N[simp]: assumes "isOK (check_coefficients_E\<^sub>I n ids fc)"
  shows "isOK (check_coefficients_N n ids fc)"
  using assms[unfolded check_coefficients_E\<^sub>I_def, simplified] by auto



definition check_coefficients_M\<^sub>I ::
 "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> showsl check"
  where
"check_coefficients_M\<^sub>I n ids fc = (do {
    check_coefficients_N n ids fc;
    check_allm (\<lambda>fx. case fx of ((f,m),(cs,c)) \<Rightarrow> (do{
        check_allm (check_M\<^sub>I n ids) cs <+? (\<lambda>str. showsl_lit(STR ''Expected all interpretation multiplicative coefficients of symbol '')
 o showsl f o showsl_lit(STR '' to be in M_I.\<newline>'') o str);
        check (\<exists>ci \<in> set(c#cs). isOK (check_M\<^sub>I n ids ci)) (
          (showsl_lit(STR ''Expected the constant coefficient '') o showsl (mat_to_list c) o
 showsl_lit(STR '' of symbol '') o showsl f o
 showsl_lit(STR '' to be in M_I whenever there are no multiplicative coefficient (i.e. symbol of arity 0).''))
        )
      })
    ) fc <+? (\<lambda>str. showsl_lit(STR ''Expected all interpretation multiplicative coeffs to be in M_I
 and the constant coeff to be in M_I whenever there are none multiplicative coeffs, for all function symbols.\<newline>'') o str)
})"

lemma check_coefficients_M\<^sub>I[simp]: assumes "isOK (check_coefficients_M\<^sub>I n ids fc)"
  shows "((f,m), (cs,c)) \<in> set fc \<Longrightarrow> set cs \<subseteq> squared_ring_mat.M\<^sub>I n (set ids) \<and>
set (c#cs) \<inter> squared_ring_mat.M\<^sub>I n (set ids) \<noteq> {}"
proof -
  assume as: "((f,m),(cs,c)) \<in> set fc"
  note check = assms[unfolded check_coefficients_M\<^sub>I_def, simplified]
  from check have "isOK (check_coefficients_N n ids fc)" by auto
  
  from check_coefficients_N[OF this as] 
  have inN: "ci \<in> set (c # cs) \<Longrightarrow> ci \<in> squared_ring_mat.N n" for ci by auto
  then have "\<forall>ci \<in> set cs. ci \<in> squared_ring_mat.M\<^sub>I n (set ids)"
            "\<exists>ci \<in> set (c#cs). ci \<in> squared_ring_mat.M\<^sub>I n (set ids)"
    by (insert as check, fastforce)+
  then show ?thesis by auto
qed

lemma check_coefficients_M\<^sub>I_N[simp]: assumes "isOK (check_coefficients_M\<^sub>I n ids fc)"
  shows "isOK (check_coefficients_N n ids fc)"
  using assms[unfolded check_coefficients_M\<^sub>I_def, simplified] by auto



subsection \<open>Fake coefficient-generator functions built from the list of function symbols and their interpretation.\<close>
text \<open>pI' is the instantiation of the pI in Linear_Poly_Interpretation.
    Outside the provided function symbols, identity matrices are given in place of useful interpretation,
    c.f. @{const default_mat_inter}.
    The identity matrix is in $E_I$, $M_I$ and also $P_I$, making it a suitable choice.\<close>
text \<open>Perform checks that are required by @{const lin_poly_inter} and @{const mono_lin_poly_inter}.\<close>

definition pI' ::
 "nat \<Rightarrow> ('f,_) fun_coeffs \<Rightarrow> ('f \<times> nat \<Rightarrow> _ mat list \<times> _ mat)"
  where
    "pI' n fc \<equiv> (\<lambda>(f,n1). case map_of fc (f,n1) of Some val \<Rightarrow> val | None \<Rightarrow> default_mat_inter n n1)"

lemma check_pI'_length[simp]: assumes "isOK (check_coeffs_length fc)"
  shows "let (cs, _) = pI' n fc f in length cs = snd f"
proof -
  obtain f' n1 where f: "f = (f',n1)" by force
  show ?thesis proof (cases "map_of fc f")
    case None
    thus ?thesis unfolding f pI'_def Let_def split by (auto simp: default_mat_inter_def)
  next
    case (Some val)
    then obtain cs c where Some: "map_of fc f = Some (cs,c)" by (cases val, auto)
    hence pI': "pI' n fc f = (cs,c)" unfolding f pI'_def Let_def by auto
    from map_of_SomeD[OF Some, unfolded f]
    have "((f', n1), cs, c) \<in> set fc" by auto
    from check_coeffs_length[OF assms this] pI' show ?thesis using f by auto
  qed
qed

lemma check_pI'_N[simp]: assumes "isOK (check_coefficients_N n ids fc)"
  shows "let (cs, c) = pI' n fc f in set (c#cs) \<subseteq> squared_ring_mat.N n"
proof -
  obtain f' n1 where f: "f = (f',n1)" by force
  show ?thesis proof (cases "map_of fc f")
    case None
    thus ?thesis unfolding f pI'_def Let_def split default_mat_inter_def 
      using squared_ring_mat.one_in_N by auto
  next
    case (Some val)
    then obtain cs c where Some: "map_of fc f = Some (cs,c)" by (cases val, auto)
    hence pI': "pI' n fc f = (cs,c)" unfolding f pI'_def Let_def by auto
    from map_of_SomeD[OF Some, unfolded f]
    have "((f', n1), cs, c) \<in> set fc" by auto
    then show ?thesis using f check_coefficients_N[OF assms] pI' by auto
  qed
qed

lemma check_pI'_E\<^sub>I[simp]: 
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
  assumes check: "isOK (check_coefficients_E\<^sub>I n ids fc)"
  shows "let (cs,c) = pI' n fc f in set cs \<subseteq> squared_ring_mat.E\<^sub>I n (set ids) \<and>
    set (c#cs) \<inter> squared_ring_mat.E\<^sub>I n (set ids) \<noteq> ({} :: 'a mat set)"
proof -
  obtain f' n1 where f: "f = (f',n1)" by force
  show ?thesis proof (cases "map_of fc f")
    case None
    thus ?thesis unfolding f pI'_def Let_def split default_mat_inter_def
      using squared_ring_mat_with_assms.one_in_E\<^sub>I[OF srm] ids
      by auto
  next
    case (Some val)
    then obtain cs c where Some: "map_of fc f = Some (cs,c)" by (cases val, auto)
    hence pI': "pI' n fc f = (cs,c)" unfolding f pI'_def Let_def by auto
    from map_of_SomeD[OF Some, unfolded f]
    have "((f', n1), cs, c) \<in> set fc" by auto
    then show ?thesis using f check_coefficients_E\<^sub>I[OF check] pI' by auto
  qed
qed


lemma check_pI'_M\<^sub>I[simp]: fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
  and check: "isOK (check_coefficients_M\<^sub>I n ids fc)"
  shows "let (cs,c) = pI' n fc f in set cs \<subseteq> squared_ring_mat.M\<^sub>I n (set ids) \<and>
    set (c#cs) \<inter> squared_ring_mat.M\<^sub>I n (set ids) \<noteq> ({} :: 'a mat set)"
proof -
  obtain f' n1 where f: "f = (f',n1)" by force
  show ?thesis proof (cases "map_of fc f")
    case None
    thus ?thesis unfolding f pI'_def Let_def split default_mat_inter_def
      using squared_ring_mat_with_assms.one_in_M\<^sub>I[OF srm] ids
      by auto
  next
    case (Some val)
    then obtain cs c where Some: "map_of fc f = Some (cs,c)" by (cases val, auto)
    hence pI': "pI' n fc f = (cs,c)" unfolding f pI'_def Let_def by auto
    from map_of_SomeD[OF Some, unfolded f]
    have "((f', n1), cs, c) \<in> set fc" by auto
    then show ?thesis using f check_coefficients_M\<^sub>I[OF check] pI' by auto
  qed
qed



definition check_coeffs_E\<^sub>I_final where
"check_coeffs_E\<^sub>I_final n ids fc = (do {
    check_coeffs_length fc;
    check_coefficients_N n ids fc;
    check_coefficients_E\<^sub>I n ids fc
})"

lemma check_coeffs_E\<^sub>I_final_triv[simp]: assumes "isOK (check_coeffs_E\<^sub>I_final n ids fc)"
  shows "isOK (check_coeffs_length fc)" "isOK ( check_coefficients_N n ids fc)"
 "isOK (check_coefficients_E\<^sub>I n ids fc)"
  using assms[unfolded check_coeffs_E\<^sub>I_final_def] by auto

lemma check_coeffs_E\<^sub>I_final[simp]: 
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
  and check: "isOK (check_coeffs_E\<^sub>I_final n ids fc)"
  shows "pI' n fc (f, n1) = (cs, c) \<Longrightarrow>
    set (c # cs) \<subseteq> squared_ring_mat.N n \<and> length cs = n1 \<and> set cs \<subseteq> squared_ring_mat.E\<^sub>I n (set ids) \<and>
    set (c#cs) \<inter> squared_ring_mat.E\<^sub>I n (set ids) \<noteq> ({} :: 'a mat set)"
proof -
  assume as: "pI' n fc (f, n1) = (cs, c)"
  have "set (c # cs) \<subseteq> squared_ring_mat.N n"
    using as check_pI'_N[OF check_coeffs_E\<^sub>I_final_triv(2)[OF check], of "(f,n1)"]
    by auto
  moreover
  have "length cs = n1"
    using as check_pI'_length[OF check_coeffs_E\<^sub>I_final_triv(1)[OF check], of n "(f, n1)"]
    by auto
  moreover
  have "set cs \<subseteq> squared_ring_mat.E\<^sub>I n (set ids)" "set (c#cs) \<inter> squared_ring_mat.E\<^sub>I n (set ids) \<noteq> {}"
    using as check_pI'_E\<^sub>I[OF srm ids check_coeffs_E\<^sub>I_final_triv(3)[OF check], of "(f,n1)"]
    by auto
  ultimately
  show ?thesis by auto
qed
  


subsection \<open>Final checks on the function symbols' interpretation coefficients\<close>
\<comment> \<open>Using both the section on coefficients being in the strict carrier and the section on
    pI'.\<close>

definition check_coeffs_M\<^sub>I_final where
"check_coeffs_M\<^sub>I_final n ids fc = (do {
    check_coeffs_length fc;
    check_coefficients_N n ids fc;
    check_coefficients_M\<^sub>I n ids fc
})"

lemma check_coeffs_M\<^sub>I_final_triv[simp]: assumes "isOK (check_coeffs_M\<^sub>I_final n ids fc)"
  shows "isOK (check_coeffs_length fc)" "isOK (check_coefficients_N n ids fc)"
 "isOK (check_coefficients_M\<^sub>I n ids fc)"
  using assms[unfolded check_coeffs_M\<^sub>I_final_def] by auto

lemma check_coeffs_M\<^sub>I_final[simp]: fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt" 
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_coeffs_M\<^sub>I_final n ids fc)"
    and as: "pI' n fc (f, n1) = (cs, c)" 
  shows "set (c # cs) \<subseteq> squared_ring_mat.N n \<and> length cs = n1 \<and> set cs \<subseteq> squared_ring_mat.M\<^sub>I n (set ids) \<and>
set (c#cs) \<inter> squared_ring_mat.M\<^sub>I n (set ids) \<noteq> ({} :: 'a mat set)"
proof -
  have "set (c # cs) \<subseteq> squared_ring_mat.N n"
    using as check_pI'_N[OF check_coeffs_M\<^sub>I_final_triv(2)[OF check], of "(f,n1)"]
    by auto
  moreover
  have "length cs = n1"
    using as check_pI'_length[OF check_coeffs_M\<^sub>I_final_triv(1)[OF check], of n "(f, n1)"]
    by auto
  moreover
  have "set cs \<subseteq> squared_ring_mat.M\<^sub>I n (set ids)" "set (c#cs) \<inter> squared_ring_mat.M\<^sub>I n (set ids) \<noteq> {}"
    using as check_pI'_M\<^sub>I[OF srm ids check_coeffs_M\<^sub>I_final_triv(3)[OF check], of "(f,n1)"]
    by auto
  ultimately
  show ?thesis by auto
qed



subsection \<open>Useless cruft section\<close>
\<comment> \<open>Proving that every single polynomial evaluation function being used by the implementation
    is the same regardless of mat_interpretation_parameter.Matsemiring being used instead of
    mat_interpretation_parameter.R_mat_semiring.\<close>

definition RringEI where "RringEI n gt ids = mat_interpretation_parameter.R_mat_semiring n gt
 (squared_ring_mat.E\<^sub>I n (set ids)) (squared_ring_mat_with_assms.core n gt (squared_ring_mat.E\<^sub>I n (set ids)))"
definition RringMI where "RringMI n gt ids = mat_interpretation_parameter.R_mat_semiring n gt
 (squared_ring_mat.M\<^sub>I n (set ids)) (squared_ring_mat_with_assms.core n gt (squared_ring_mat.M\<^sub>I n (set ids)))"
definition Rring where "Rring n = mat_interpretation_parameter.Matsemiring n (\<lambda> _ _. True)"


definition RringSet where "RringSet n gt ids S U =
 mat_interpretation_parameter.R_mat_semiring n gt (S n ids) (U n ids)"

lemma RringEI_to_Set: "RringEI n gt ids = RringSet n gt ids 
  (\<lambda> n ids. squared_ring_mat.E\<^sub>I n (set ids))
  (\<lambda> n ids. squared_ring_mat_with_assms.core n gt (squared_ring_mat.E\<^sub>I n (set ids)))" 
  unfolding RringEI_def RringSet_def by simp

lemma RringMI_to_Set: "RringMI n gt ids = RringSet n gt ids 
  (\<lambda> n ids. squared_ring_mat.M\<^sub>I n (set ids))
  (\<lambda> n ids. squared_ring_mat_with_assms.core n gt (squared_ring_mat.M\<^sub>I n (set ids)))" 
  unfolding RringMI_def RringSet_def by simp

lemmas mat_ring_defs = 
  RringSet_def Rring_def
  mat_interpretation_parameter.R_mat_semiring_def
  mat_interpretation_parameter.Matsemiring_def

lemma vpoly_eq: "vpoly (RringSet n gt ids S U) a = vpoly (Rring n) a"
  unfolding vpoly_def RringSet_def Rring_def
            mat_interpretation_parameter.R_mat_semiring_def
            mat_interpretation_parameter.Matsemiring_def
  by auto

lemma add_var_eq: "add_var (RringSet n gt ids S U) x a xs = add_var (Rring n) x a xs"
  unfolding add_var_def add_var_sumC_def add_var_graph_def
  unfolding mat_ring_defs by simp
  
lemma sum_pvars_eq: "sum_pvars (RringSet n gt ids S U) vas vbs = sum_pvars (Rring n) vas vbs"
  unfolding sum_pvars_def sum_pvars_sumC_def sum_pvars_graph_def
  unfolding add_var_eq
  unfolding mat_ring_defs by simp

lemma sum_lpoly_eq: "sum_lpoly (RringSet n gt ids S U) p q = sum_lpoly (Rring n) p q"
  unfolding sum_lpoly_def sum_lpoly_sumC_def sum_lpoly_graph_def
  unfolding sum_pvars_eq 
  unfolding mat_ring_defs by simp

lemma list_sum_poly_eq: "list_sum_poly (RringSet n gt ids S U) xs = list_sum_poly (Rring n) xs"
  unfolding list_sum_poly_def list_sum_poly_sumC_def list_sum_poly_graph_def
  unfolding sum_lpoly_eq 
  unfolding mat_ring_defs by simp

lemma mul_pvars_eq: "mul_pvars (RringSet n gt ids S U) a p = mul_pvars (Rring n) a p"
  unfolding mul_pvars_def mul_pvars_sumC_def mul_pvars_graph_def
  unfolding mat_ring_defs by simp

lemma mul_lpoly_eq: "mul_lpoly (RringSet n gt ids S U) a p = mul_lpoly (Rring n) a p"
  unfolding mul_lpoly_def mul_lpoly_sumC_def mul_lpoly_graph_def
  unfolding mul_pvars_eq
  unfolding mat_ring_defs by simp

lemma evalp_eq: "evalp (RringSet n gt ids S U) p t = evalp (Rring n) p t"
proof (induction t)
  case (Var x)
  then show ?case unfolding Term.eval_term.simps by (rule vpoly_eq)
next
  case (Fun f xs)
  then show ?case
  proof -
    let ?R1 = "RringSet n gt ids S U"
    let ?R2 = "Rring n"
    have "evalp ?R1 p (Fun f xs) = Ip ?R1 p f [evalp ?R1 p s. s \<leftarrow> xs]"
      by auto
    also have "\<dots> = Ip ?R1 p f [evalp ?R2 p s. s \<leftarrow> xs]"
    proof -
      have "[evalp ?R1 p s. s \<leftarrow> xs] = [evalp ?R2 p s. s \<leftarrow> xs]"
        using Fun.IH by auto
      then show ?thesis by argo
    qed
    also have "\<dots> = Ip ?R2 p f [evalp ?R2 p s. s \<leftarrow> xs]"
    proof (cases "p (f, length [evalp ?R2 p s. s \<leftarrow> xs])")
      case (Pair cs c)
      then show ?thesis
      proof -
        define as where "as = [evalp ?R2 p s. s \<leftarrow> xs]"
        have "Ip ?R1 p f as =
          list_sum_poly ?R1 (c_lpoly c # map (\<lambda> ca. mul_lpoly ?R1 (fst ca) (snd ca))
             (zip cs as))"
          unfolding Ip_def as_def
          using Pair
          by auto
        also have "\<dots> = list_sum_poly ?R1 (c_lpoly c # map (\<lambda> ca. mul_lpoly ?R2 (fst ca) (snd ca))
             (zip cs as))"
          using mul_lpoly_eq
          by metis
        also have "\<dots> = list_sum_poly ?R2 (c_lpoly c # map (\<lambda> ca. mul_lpoly ?R2 (fst ca) (snd ca))
             (zip cs as))"
          using list_sum_poly_eq[of n gt ids S U]
          by blast
        also have "\<dots> = Ip ?R2 p f as"
          unfolding Ip_def as_def
          using Pair
          by auto
        finally show ?thesis unfolding as_def.
      qed
    qed
    also have "\<dots> = evalp ?R2 p (Fun f xs)"
      by auto
    finally show ?thesis.
  qed
qed

lemma sub_var_eq: "sub_var (RringSet n gt ids S U) a b c = sub_var (Rring n) a b c"
  unfolding sub_var_def sub_var_sumC_def sub_var_graph_def expl_a_minus_def
  unfolding mat_ring_defs by simp

lemma sub_pvars_eq: "sub_pvars (RringSet n gt ids S U) a b = sub_pvars (Rring n) a b"
  unfolding sub_pvars_def sub_pvars_sumC_def sub_pvars_graph_def sub_var_eq
  unfolding mat_ring_defs by simp

lemma sub_lpoly_eq: "sub_lpoly (RringSet n gt ids S U) p q = sub_lpoly (Rring n) p q"
  unfolding sub_lpoly_def sub_lpoly_sumC_def sub_lpoly_graph_def
  unfolding sub_pvars_eq expl_a_minus_def
  unfolding mat_ring_defs by simp
  
lemma evp1 [simp]: "evalp_rule (RringEI n gt ids) p l r = evalp_rule (Rring n) p l r"
  unfolding evalp_rule_def 
  unfolding RringEI_to_Set 
  unfolding sub_lpoly_eq evalp_eq ..

lemma evp2[simp]: "evalp_rule (RringMI n gt ids) p l r = evalp_rule (Rring n) p l r"
  unfolding evalp_rule_def
  unfolding RringMI_to_Set 
  unfolding sub_lpoly_eq evalp_eq ..


subsection \<open>Checks on TRS rules\<close>
\<comment> \<open>After working on variable interpretations and function symbol interpretations,
    now working on TRS rules being in the monotonic carrier (see carrierMono in Linear_Poly_Interpretation).\<close>
\<comment> \<open>Checks on:
    All interpretations of rule of a TRS being in the positive cone (>= 0),
    All interpretations of rule of a TRS we want to prove terminating being in the monotonic carrier.\<close>
\<comment> \<open>To be in the monotonic carrier, it suffices that all coefficients of the polynomial interpretation
    be in the positive cone and that at least 1 coefficient be in the monotonic carrier.\<close>


definition check_lpoly_coef_N where
"check_lpoly_coef_N n lp =
    check_allm (check_N n ) (coeffs_of_lpoly_better lp) <+? (\<lambda>str.
      showsl_lit (STR ''Expected all lpoly coefficient to be greater or equal to 0 mat.\<newline>'') o str)"

lemma check_lpoly_coef_N[simp]: assumes "isOK (check_lpoly_coef_N n lp)"
  shows "set (coeffs_of_lpoly_better lp) \<subseteq> squared_ring_mat.N n"
  using assms[unfolded check_lpoly_coef_N_def, simplified] by auto


definition check_poly_coef_one_P\<^sub>I where
"check_poly_coef_one_P\<^sub>I n gt ids lp = do {
    check_lpoly_coef_N n lp;
    check (\<exists>c \<in> set (coeffs_of_lpoly_better lp). isOK (check_P\<^sub>I n gt ids c))
      (showsl_lit (STR ''Expected at least one lpoly coefficient to be in P_I, got none.'')
      o showsl_list (map mat_to_list (coeffs_of_lpoly_better lp)))}"

lemma check_poly_coef_one_P\<^sub>I[simp]: assumes "isOK (check_poly_coef_one_P\<^sub>I n gt ids lp)"
  shows "set (coeffs_of_lpoly_better lp) \<inter> squared_ring_mat.P\<^sub>I n gt (set ids) \<noteq> {}"
  using assms[unfolded check_poly_coef_one_P\<^sub>I_def, simplified] check_lpoly_coef_N by fastforce


definition check_poly_coef_one_L\<^sub>I where
"check_poly_coef_one_L\<^sub>I n gt ids lp = do {
    check_L\<^sub>I n gt ids (coeffs_of_lpoly_better lp)
    } <+? (\<lambda> s. showsl_lit (STR ''problem in checking strict L_I-decrease.\<newline>'') o s)"

definition check_rules_N_E\<^sub>I :: "nat \<Rightarrow> nat list \<Rightarrow> ('f ::showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check" where
"check_rules_N_E\<^sub>I n ids fc R =
    check_allm (\<lambda>(l,r). (check_lpoly_coef_N n (
              evalp_rule (Rring n) (pI' n fc) l r
            )
      ) <+? (\<lambda>str. showsl_lit(STR ''Expected rule interpretation of lhs '') o showsl l o showsl_lit(STR '' and of rhs '')
 o showsl r o showsl_lit (STR '' to be in N.\<newline>'') o str)
    ) R <+? (\<lambda>str.
    (showsl_lit (STR ''Expected all interpretation of rules (i.e. I l - I r for all rule l -> r)
 to be greater or equal to 0 mat.\<newline>'')) o str)"

lemma check_rules_N_E\<^sub>I[simp]: assumes "isOK (check_rules_N_E\<^sub>I n ids fc R)"
  shows "(l,r) \<in> set R \<Longrightarrow> set (coeffs_of_lpoly_better (
  evalp_rule (Rring n) (pI' n fc) l r 
)) \<subseteq> squared_ring_mat.N n"
  using assms[unfolded check_rules_N_E\<^sub>I_def, simplified] check_lpoly_coef_N[of n]
  by blast

definition check_rules_N_M\<^sub>I :: "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check" where
"check_rules_N_M\<^sub>I n ids fc R =
    check_allm (\<lambda>(l,r). (check_lpoly_coef_N n (
              evalp_rule (Rring n) (pI' n fc) l r
            )
      ) <+? (\<lambda>str. showsl_lit(STR ''Expected rule interpretation of lhs '') o showsl l o showsl_lit(STR '' and of rhs '')
 o showsl r o showsl_lit (STR '' to be in N. Thus, expected all poly coeffs to be in N:\<newline>'')
 o showsl (evalp_rule (Rring n) (pI' n fc) l r) o str)
    ) R <+? (\<lambda>str.
    (showsl_lit (STR ''Expected all interpretation of rules (i.e. ids l - ids r for all rule l -> r)
 to be greater or equal to 0 mat.\<newline>'')) o str)"

lemma check_rules_N_M\<^sub>I[simp]: assumes "isOK (check_rules_N_M\<^sub>I n ids fc R)"
  shows "(l,r) \<in> set R \<Longrightarrow> set (coeffs_of_lpoly_better (
  evalp_rule (Rring n) (pI' n fc) l r 
)) \<subseteq> squared_ring_mat.N n"
  using assms[unfolded check_rules_N_M\<^sub>I_def, simplified] check_lpoly_coef_N[of n]
  by blast



definition check_rules_one_P\<^sub>I ::
 "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check" where
"check_rules_one_P\<^sub>I n gt ids fc R =
    check_allm (\<lambda>(l,r). (check_poly_coef_one_P\<^sub>I n gt ids (
              evalp_rule (Rring n) (pI' n fc) l r
            )
      ) <+? (\<lambda>str. showsl_lit(STR ''Expected rule interpretation of lhs '') o showsl l o showsl_lit(STR '' and of rhs '')
 o showsl r o showsl_lit (STR '' to be in P_I. Thus, expected at least 1 poly coeffs to be in P_I:\<newline>'')
 o showsl (evalp_rule (Rring n) (pI' n fc) l r) o str)
    ) R <+? (\<lambda>str.
    (showsl_lit (STR ''Expected all interpretation of rules (i.e. ids l - ids r for all rule l -> r)
 to have at least one lpoly coef in P_I.\<newline>'')) o str)"

lemma check_rules_one_P\<^sub>I[simp]: assumes "isOK (check_rules_one_P\<^sub>I n gt ids fc R)"
  shows "(l,r) \<in> set R \<Longrightarrow> set (coeffs_of_lpoly_better (

  evalp_rule (Rring n) (pI' n fc) l r 

)) \<inter> squared_ring_mat.P\<^sub>I n gt (set ids) \<noteq> {}"
  using assms[unfolded check_rules_one_P\<^sub>I_def] by auto

definition check_rules_one_L\<^sub>I ::
 "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> ('f :: showl, _) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check"
 where
"check_rules_one_L\<^sub>I n gt ids fc R =
    check_allm (\<lambda>(l,r). (check_poly_coef_one_L\<^sub>I n gt ids (
              evalp_rule (Rring n) (pI' n fc) l r
            )
      ) <+? (\<lambda>str. showsl_lit(STR ''Expected rule interpretation of lhs '') o showsl l o showsl_lit(STR '' and of rhs '')
 o showsl r o showsl_lit (STR '' to be in L_I. Thus, expected L_I decrease in each indexed row:\<newline>'')
 o showsl (evalp_rule (Rring n) (pI' n fc) l r) o str)
    ) R <+? (\<lambda>str.
    (showsl_lit (STR ''Expected all interpretation of rules (i.e. ids l - ids r for all rule l -> r)
 to have at least one lpoly coef in L_I.\<newline>'')) o str)"


subsection \<open>(relative) termination checks for domain = E_I\<close>
\<comment> \<open>Checks that the rule of a TRS are in the monotonic carrier
   (that is the core of the strict carrier, see Matrix_Base).
    It mechanically implies termination.\<close>
\<comment> \<open>For termination relative to a TRS S, it suffices that the interpretations of the rule of S
    be in the positive cone, in addition to the above requirements.\<close>


definition check_core_E\<^sub>I :: "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check" where
"check_core_E\<^sub>I n gt ids fc R = (do {
    check_coeffs_E\<^sub>I_final n ids fc;
    check_rules_N_E\<^sub>I n ids fc R;
    check_rules_one_P\<^sub>I n gt ids fc R
})"

lemma check_core_E\<^sub>I_triv[simp]: assumes "isOK (check_core_E\<^sub>I n gt ids fc R)"
  shows "isOK (check_coeffs_E\<^sub>I_final n ids fc)"
        "isOK (check_rules_N_E\<^sub>I n ids fc R)"
        "isOK (check_rules_one_P\<^sub>I n gt ids fc R)"
  using assms[unfolded check_core_E\<^sub>I_def] by auto


lemma check_core_E\<^sub>I:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
  and check: "isOK (check_core_E\<^sub>I n gt ids fc R)"
  and "range \<alpha> \<subseteq> squared_ring_mat.E\<^sub>I n (set ids)" 
  shows "
 (squared_ring_mat_with_assms.eval_rule n gt (squared_ring_mat.E\<^sub>I n (set ids)) (pI' n fc) \<alpha>) ` (set R) \<subseteq>
   squared_ring_mat_with_assms.core n gt (squared_ring_mat.E\<^sub>I n (set ids))"
proof -
  interpret srm: squared_ring_mat_with_assms mo n def gt by fact
  from ids 
  have t2: "n > 0" by auto
  have t1: "mat_interpretation_parameter.wf_carrierS n gt (srm.E\<^sub>I (set ids))" by (rule E\<^sub>I_wf_carrierS[OF srm ids(2,1)])
  let ?pi = "pI' n fc" 
  have "isOK (check_coeffs_E\<^sub>I_final n ids fc)" using check by auto
  from check_coeffs_E\<^sub>I_final[OF srm ids this] have 
    t0: "\<And>f na cs c. ?pi (f, na) = (cs, c) \<Longrightarrow> set (c # cs) \<subseteq> srm.N \<and> length cs = na \<and> set (c # cs) \<inter> srm.E\<^sub>I (set ids) \<noteq> {}" by auto
  note strict_def = srm.strict_check_def[of ?pi, OF t0 t2 t1, of "\<lambda> x _ _ _. x"]
  note th = srm.strict_check_in_mono_imp_forall_assig_trs[OF t0 t2 t1 _ srm.valid_strict_criterion_None, 
      of ?pi "\<lambda> x _ _ _ . x" ?pi "set R", folded RringEI_def] 
  
  show ?thesis 
  proof ((rule th; (intro assms t1 t2)?), goal_cases)
    case (4 l r)
    have "srm.strict_check (srm.E\<^sub>I (set ids)) None (evalp_rule (srm.current_semiring (srm.E\<^sub>I (set ids))) (pI' n fc) l r)" 
      using check_rules_one_P\<^sub>I[OF check_core_E\<^sub>I_triv(3)[OF check] 4]  
      apply (subst strict_def, force)
      apply simp
      apply (unfold RringEI_def[symmetric])
      unfolding evp1 
      using squared_ring_mat_with_assms.core_E\<^sub>I_is_P\<^sub>I[OF srm]
      using ids by blast
    thus ?case 
      using check_rules_N_E\<^sub>I[OF check_core_E\<^sub>I_triv(2)[OF check] 4]
      apply (unfold RringEI_def[symmetric])
      unfolding evp1 
      by auto
  qed (insert check_coeffs_E\<^sub>I_final[OF srm ids check_core_E\<^sub>I_triv(1)[OF check]], auto)
qed


definition check_N_for_relative_E\<^sub>I ::
 "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check"
 where
"check_N_for_relative_E\<^sub>I n ids fc R = (do {
    check_coeffs_E\<^sub>I_final n ids fc;
    check_rules_N_E\<^sub>I n ids fc R
})"

lemma check_N_for_relative_E\<^sub>I_triv[simp]: assumes "isOK (check_N_for_relative_E\<^sub>I n ids fc R)"
  shows "isOK (check_coeffs_E\<^sub>I_final n ids fc)"
        "isOK (check_rules_N_E\<^sub>I n ids fc R)"
  using assms[unfolded check_N_for_relative_E\<^sub>I_def] by auto

lemma check_N_for_relative_E\<^sub>I:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_N_for_relative_E\<^sub>I n ids fc S)"
    and ran: "range \<alpha> \<subseteq> squared_ring_mat.E\<^sub>I n (set ids)" 
  shows "(squared_ring_mat_with_assms.eval_rule n gt (squared_ring_mat.E\<^sub>I n (set ids)) (pI' n fc) \<alpha>) ` (set S) \<subseteq>
 squared_ring_mat.N n"
proof -
  from ids have t2: "n > 0" by auto

  note th = squared_ring_mat_with_assms.all_coeff_in_N_imp_forall_assig_trs[
      OF srm _ t2 E\<^sub>I_wf_carrierS[OF srm], OF _ ids(2,1), folded RringEI_def]
  have orient: "(\<And>l r. (l, r) \<in> set S \<Longrightarrow> set (coeffs_of_lpoly_better
   (evalp_rule (RringEI n gt ids) (pI' n fc) l r))
          \<subseteq> squared_ring_mat.N n)"
    using check_rules_N_E\<^sub>I[OF check_N_for_relative_E\<^sub>I_triv(2)[OF check]]
    by auto
  show ?thesis
    by (rule th[OF _ _ orient ran], 
        insert check_coeffs_E\<^sub>I_final[OF srm ids check_N_for_relative_E\<^sub>I_triv(1)[OF check]] ids, auto)
qed



subsection \<open>(relative) termination checks for domain = M_I\<close>
\<comment> \<open>Same comments than for domain = E_I\<close>


definition check_core_M\<^sub>I :: "nat \<Rightarrow> _ \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check" where
"check_core_M\<^sub>I n gt ids fc R = (do {
    check_coeffs_M\<^sub>I_final n ids fc;
    check_rules_N_M\<^sub>I n ids fc R;
    check_rules_one_L\<^sub>I n gt ids fc R
})"

lemma check_core_M\<^sub>I_triv[simp]: assumes "isOK (check_core_M\<^sub>I n gt ids fc R)"
  shows "isOK (check_coeffs_M\<^sub>I_final n ids fc)"
        "isOK (check_rules_N_M\<^sub>I n ids fc R)"
        "isOK (check_rules_one_L\<^sub>I n gt ids fc R)"
  using assms[unfolded check_core_M\<^sub>I_def] by auto

lemma check_core_M\<^sub>I:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
    and R :: "('f :: showl, 'v :: showl)rules" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_core_M\<^sub>I n gt ids fc R)"
    and ran: "range \<alpha> \<subseteq> squared_ring_mat.M\<^sub>I n (set ids)" 
  shows "(squared_ring_mat_with_assms.eval_rule n gt (squared_ring_mat.M\<^sub>I n (set ids)) (pI' n fc) \<alpha>) ` (set R) \<subseteq>
 squared_ring_mat_with_assms.core n gt (squared_ring_mat.M\<^sub>I n (set ids))"
proof -
  interpret srm: squared_ring_mat_with_assms mo n def gt by fact
  from ids 
  have t2: "n > 0" by auto
  have t1: "mat_interpretation_parameter.wf_carrierS n gt (srm.M\<^sub>I (set ids))" by (rule M\<^sub>I_wf_carrierS[OF srm ids(2,1)])
  let ?pi = "pI' n fc" 
  have "isOK (check_coeffs_M\<^sub>I_final n ids fc)" using check by auto
  from check_coeffs_M\<^sub>I_final[OF srm ids this] have 
    t0: "\<And>f na cs c. ?pi (f, na) = (cs, c) \<Longrightarrow> set (c # cs) \<subseteq> srm.N \<and> length cs = na \<and> set (c # cs) \<inter> srm.M\<^sub>I (set ids) \<noteq> {}" by auto
  from ids have t2: "n > 0" by auto
  note strict_def = srm.strict_check_def[OF t0 t2 t1, of ?pi "\<lambda> x _ _ _. x"]

  define crit where "crit = (\<lambda> (p :: ('v,'a mat)l_poly). \<forall> i \<in> set ids. \<exists> c \<in> set (coeffs_of_lpoly_better p). \<exists> j \<in> set ids. gt (c $$ (i,j)) 0)" 
  

  note th = srm.strict_check_in_mono_imp_forall_assig_trs[OF _ t2 t1, of ?pi "Some crit" "set R" \<alpha>,
      folded RringMI_def, OF t0 _ _ _ ran, of "\<lambda> x _ _ _. x"]
  show ?thesis
  proof (rule th, blast, goal_cases)
    case 2
    {
      fix \<alpha> :: "'v \<Rightarrow> 'a mat" and p
      assume ran: "range \<alpha> \<subseteq> srm.M\<^sub>I (set ids)" 
        and coeffsN: "set (coeffs_of_lpoly_better p) \<subseteq> srm.N" 
        and crit: "crit p" 
      let ?ring = "srm.current_semiring (srm.M\<^sub>I (set ids))" 
      let ?eval = "semiring.eval_lpoly ?ring \<alpha>" 
      obtain a vas where p: "p = LPoly a vas" by (cases p, auto)
      from coeffsN[unfolded p] have coeffs: "a \<in> srm.N" "snd ` set vas \<subseteq> srm.N" 
        by (auto simp: coeffs_of_pvars_better_def)
      interpret lpi_mat: lin_poly_inter ?ring ?pi
        by (rule srm.lpi_mat[OF t0 t2 t1], auto)
      have "?eval p = a + lpi_mat.eval_pvars \<alpha> vas" 
        unfolding p lpi_mat.eval_lpoly.simps 
        by (simp add: mat_ring_defs)
      also have "lpi_mat.eval_pvars \<alpha> vas = squared_ring_mat.add_list_mat n (map (\<lambda> (v,ai). ai * \<alpha> v) vas)" 
      proof (induct vas)
        case Nil
        show ?case by (simp, simp add: mat_ring_defs)
      next
        case (Cons va vas)
        obtain v a where va: "va = (v,a)" by force
        show ?case unfolding va
          by (simp add: Cons, simp add: mat_ring_defs)
      qed
      finally have eval: "?eval p = a + srm.add_list_mat (map (\<lambda>(v, ai). ai * \<alpha> v) vas)" .

      from ran have ranN: "range \<alpha> \<subseteq> srm.N" 
        using srm.M\<^sub>I_def by auto 
      hence ranC: "\<alpha> x \<in> carrier_mat n n" for x unfolding srm.N_def by auto

      have sumN: "srm.add_list_mat (map (\<lambda>(v, ai). ai * \<alpha> v) vas) \<in> srm.N" 
      proof (intro srm.add_list_mat_closed srm.N_is_submonoid_add)
        show "set (map (\<lambda>(v, ai). ai * \<alpha> v) vas) \<subseteq> srm.N" using coeffs ranN 
          by (auto intro!: srm.N_mult_closed)
      qed
      hence sumC: "srm.add_list_mat (map (\<lambda>(v, ai). ai * \<alpha> v) vas) \<in> carrier_mat n n" unfolding srm.N_def by auto
      from coeffs have aC: "a \<in> carrier_mat n n" by (auto simp: srm.N_def)

      have "?eval p \<in> srm.L\<^sub>I (set ids)" unfolding srm.L\<^sub>I_def eval
      proof (standard, intro conjI ballI srm.N_add_closed coeffs sumN)  
        fix i
        assume i: "i \<in> set ids" 
        from crit[unfolded crit_def p, rule_format, OF this]
        obtain j c where j: "j \<in> set ids" and c: "c \<in> insert a (snd ` set vas)" and gt: "gt (c $$ (i,j)) 0" 
          by (auto simp: coeffs_of_pvars_better_def)
        from i j ids have ij: "i < n" "j < n" by auto
        show "\<exists> j \<in> set ids. gt ((a + srm.add_list_mat (map (\<lambda>(v, ai). ai * \<alpha> v) vas)) $$ (i, j)) 0" 
        proof -
          define list where "list = (\<lambda> j. map (\<lambda>(v, ai). (ai * \<alpha> v) $$ (i,j)) vas)" 
          {
            fix k
            assume k: "k \<in> set ids" 
            with ij ids have ik: "i < n" "k < n" by auto
            have "set ms \<subseteq> carrier_mat n n \<Longrightarrow> srm.add_list_mat ms \<in> carrier_mat n n \<and> srm.add_list_mat ms $$ (i,k) = sum_list (map (\<lambda> m. m $$ (i,k)) ms)" for ms
            proof (induct ms)
              case Nil
              thus ?case using ik by auto
            next
              case (Cons m ms)
              thus ?case using ik by auto
            qed
            hence add_list_sum: "set ms \<subseteq> carrier_mat n n \<Longrightarrow> srm.add_list_mat ms $$ (i,k) = sum_list (map (\<lambda> m. m $$ (i,k)) ms)" for ms by auto

            have id: "(a + srm.add_list_mat (map (\<lambda>(v, ai). ai * \<alpha> v) vas)) $$ (i, k)
              = a $$ (i,k) + sum_list (list k)" 
              unfolding srm.mat_add_compo_alt[OF aC sumC ik] list_def
              apply (intro arg_cong[of _ _ "(+) _"])
              apply (subst add_list_sum)
              subgoal using coeffs ranN by (auto simp: srm.N_def ranC intro!: mult_carrier_mat) 
              by (induct vas, auto)

            {
              fix x b
              assume "(x,b) \<in> set vas"
              with coeffs have b: "b \<in> srm.N" by auto
              from ranN have "\<alpha> x \<in> srm.N" by auto
              with b have "(b * \<alpha> x) \<in> srm.N" by (rule srm.N_mult_closed)
              hence "0 \<le> (b * \<alpha> x) $$ (i,k)" using ik unfolding srm.N_def by auto
            }
            hence listN: "y \<in> set (list k) \<Longrightarrow> y \<ge> 0" for y unfolding list_def by auto
            note id listN
          } note common_steps = this

          show ?thesis
          proof (cases "c = a")
            case True
            with gt have gt: "gt (a $$ (i,j)) 0" by auto
            have "sum_list (list j) \<ge> 0" 
              by (rule sum_list_ge_0, insert common_steps(2)[OF j], auto)
            with gt have "gt (a $$ (i,j) + sum_list (list j)) 0" 
              by (meson add_increasing2 ge_refl srm.compat)
            thus ?thesis using common_steps(1)[OF j] j by metis
          next
            case False
            with c obtain x where xc: "(x,c) \<in> set vas" by force
            from coeffs xc have cN: "c \<in> srm.N" by auto
            hence c: "c \<in> carrier_mat n n" unfolding srm.N_def by auto
            from ran have "\<alpha> x \<in> srm.M\<^sub>I (set ids)" by auto
            from this[unfolded srm.M\<^sub>I_def srm.N_def] j obtain k 
              where \<alpha>: "\<alpha> x \<in> carrier_mat n n" and kI: "k \<in> set ids" and ge1: "\<alpha> x $$ (j,k) \<ge> 1" by auto
            from kI ids have k: "k < n" by auto
            from xc have "(c * \<alpha> x) $$ (i,k) \<in> set (list k)" unfolding list_def by auto
            from split_list[OF this] obtain l1 l2 where list: "list k = l1 @ (c * \<alpha> x) $$ (i,k) # l2" by auto
            define b where "b = a $$ (i,k) + sum_list (l1 @ l2)" 
            have id: "a $$ (i,k) + sum_list (list k) = (c * \<alpha> x) $$ (i,k) + b" unfolding b_def list by auto
            from coeffs ij k have aN: "a $$ (i,k) \<ge> 0" unfolding srm.N_def by auto

            have b0: "b \<ge> 0" unfolding b_def 
              by (intro sum_list_ge_0 add_nonneg_nonneg aN, intro common_steps(2)[OF kI], auto simp: list)

            define d where "d = (\<Sum>l \<in> {0..<n} - {j}. c $$ (i, l) * \<alpha> x $$ (l, k))"
            have d0: "d \<ge> 0" unfolding d_def 
              apply (intro sum_nonneg mult_nonneg_nonneg)
              subgoal using cN ij(1) unfolding srm.N_def by auto
              subgoal using ranN k unfolding srm.N_def by force
              done

            define e where "e = c $$ (i,j) * \<alpha> x $$ (j, k)"
            from ge1 gt have e: "gt e 0" unfolding e_def 
              by (metis (no_types, lifting) mult_1_right mult_left_mono srm.compat srm.gt_imp_ge)
            
            have "(c * \<alpha> x) $$ (i,k) = (\<Sum>l = 0..<n. c $$ (i, l) * \<alpha> x $$ (l, k))" 
              using srm.mat_mult_compo_alt[OF c \<alpha> ij(1) k] by auto
            also have "\<dots> = e + d"  unfolding d_def e_def
              using ij(2) by (simp add: sum.remove)
            finally have "gt ((c * \<alpha> x) $$ (i,k)) 0" using e d0
              by (metis Groups.add_ac(2) group_cancel.rule0 srm.compat2 srm.plus_gt_right_mono)

            hence "gt (a $$ (i,k) + sum_list (list k)) 0" unfolding id using b0 
              by (metis Groups.add_ac(2) group_cancel.rule0 srm.compat2 srm.plus_gt_right_mono)

            thus ?thesis using common_steps(1)[OF kI] kI by metis
          qed
        qed
      qed
      hence "semiring.eval_lpoly (RringMI n gt ids) \<alpha> p \<in> srm.L\<^sub>I (set ids)" 
        unfolding RringMI_def . 
      with srm.core_M\<^sub>I_includes_L\<^sub>I[OF ids(2,1)] 
      have "semiring.eval_lpoly (RringMI n gt ids) \<alpha> p \<in> srm.core (srm.M\<^sub>I (set ids))" by blast
    }

    thus ?case by (subst srm.valid_strict_criterion_def[OF t0 t2 t1], force)
        (auto simp: RringMI_def)
  next
    case (3 l r)
    show ?case
    proof (intro conjI, goal_cases)
      case 1
      show ?case using check_rules_N_M\<^sub>I[OF check_core_M\<^sub>I_triv(2)[OF check] 3]
        unfolding evp2 by auto
    next
      case 2
      from check[unfolded check_core_M\<^sub>I_def]
      have "isOK(check_rules_one_L\<^sub>I n gt ids fc R)" by auto
      from this[unfolded check_rules_one_L\<^sub>I_def] 3 
      have "isOK (check_poly_coef_one_L\<^sub>I n gt ids (evalp_rule (Rring n) (pI' n fc) l r))" by auto
      from this[unfolded check_poly_coef_one_L\<^sub>I_def check_L\<^sub>I_def, simplified]
      have "crit (evalp_rule (Rring n) (pI' n fc) l r)"
        unfolding crit_def by auto
      thus ?case
        by (subst strict_def, force) auto
    qed
  qed (insert check_coeffs_M\<^sub>I_final[OF srm ids check_core_M\<^sub>I_triv(1)[OF check]], auto)
qed


definition check_N_for_relative_M\<^sub>I ::
 "nat \<Rightarrow> nat list \<Rightarrow> ('f :: showl,_) fun_coeffs \<Rightarrow> ('f, 'v :: showl) rules \<Rightarrow> showsl check"
 where
"check_N_for_relative_M\<^sub>I n ids fc R = (do {
    check_coeffs_M\<^sub>I_final n ids fc;
    check_rules_N_M\<^sub>I n ids fc R
})"

lemma check_N_for_relative_M\<^sub>I_triv[simp]: assumes "isOK (check_N_for_relative_M\<^sub>I n ids fc R)"
  shows "isOK (check_coeffs_M\<^sub>I_final n ids fc)"
        "isOK (check_rules_N_M\<^sub>I n ids fc R)"
  using assms[unfolded check_N_for_relative_M\<^sub>I_def] by auto


lemma check_N_for_relative_M\<^sub>I:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_N_for_relative_M\<^sub>I n ids fc S)"
    and ran: "range \<alpha> \<subseteq> squared_ring_mat.M\<^sub>I n (set ids)" 
  shows "(squared_ring_mat_with_assms.eval_rule n gt (squared_ring_mat.M\<^sub>I n (set ids)) (pI' n fc) \<alpha>) ` (set S) \<subseteq>
 squared_ring_mat.N n"
proof -
  from ids have t2: "n > 0" by auto

  note th = squared_ring_mat_with_assms.all_coeff_in_N_imp_forall_assig_trs[OF 
      srm _ t2 M\<^sub>I_wf_carrierS[OF srm ids(2,1)], folded RringMI_def]
  have orient: "(\<And>l r. (l, r) \<in> set S \<Longrightarrow> set (coeffs_of_lpoly_better
   (evalp_rule (RringMI n gt ids) (pI' n fc) l r))
          \<subseteq> squared_ring_mat.N n)"
    using check_rules_N_M\<^sub>I[OF check_N_for_relative_M\<^sub>I_triv(2)[OF check]]
    by auto
  show ?thesis
    by (rule th[OF _ _ orient ran],
      insert check_coeffs_M\<^sub>I_final[OF srm ids check_N_for_relative_M\<^sub>I_triv(1)[OF check]], auto)
qed



lemma check_M\<^sub>I_to_redtriple_order:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_core_M\<^sub>I n gt ids fc R)" "isOK (check_N_for_relative_M\<^sub>I n ids fc S)"
  shows "\<exists> s ns. mono_redtriple_order s ns ns \<and> set R \<subseteq> s \<and> set S \<subseteq> ns \<and> ce_compatible s"
proof -
  from ids have t2: "n > 0" by auto
  have check_final: "isOK (check_coeffs_M\<^sub>I_final n ids fc)" 
    using assms by auto
  have def: "(\<And>c m. (c, m) \<notin> set (map fst fc) \<Longrightarrow> pI' n fc (c, m) = default_mat_inter n m)" 
    unfolding pI'_def split 
    by (auto split: option.splits dest: map_of_SomeD) 

  note th = squared_ring_mat_with_assms.A_interpretation_to_redtriple_orientation[OF srm 
     _ t2 M\<^sub>I_wf_carrierS[OF srm ids(2,1)],
      of "pI' n fc" _ _ "map fst fc", OF _ _ _ _ def]
  show ?thesis
  proof ((rule th; (intro allI impI)?), goal_cases)
    case (3 \<alpha>)
    show ?case using check_core_M\<^sub>I[OF srm ids check(1) 3] .
  next
    case (4 \<alpha>)
    show ?case using check_N_for_relative_M\<^sub>I[OF srm ids check(2) 4] .
  qed (insert check_coeffs_M\<^sub>I_final[OF srm ids check_final], auto)
qed


lemma check_E\<^sub>I_to_redtriple_order:
  fixes gt ::
    "'a :: {Rings.ordered_semiring_0,ring_1,ordered_semiring_1,showl} \<Rightarrow> 'a \<Rightarrow> bool" 
  assumes srm: "squared_ring_mat_with_assms mo def gt"
    and ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}"
    and check: "isOK (check_core_E\<^sub>I n gt ids fc R)" "isOK (check_N_for_relative_E\<^sub>I n ids fc S)"
  shows "\<exists> s ns. mono_redtriple_order s ns ns \<and> set R \<subseteq> s \<and> set S \<subseteq> ns \<and> ce_compatible s"
proof -
  from ids have t2: "n > 0" by auto

  have check_final: "isOK (check_coeffs_E\<^sub>I_final n ids fc)" 
    using assms by auto

  have def: "(\<And>c m. (c, m) \<notin> set (map fst fc) \<Longrightarrow> pI' n fc (c, m) = default_mat_inter n m)" 
    unfolding pI'_def split 
    by (auto split: option.splits dest: map_of_SomeD) 

  note th = squared_ring_mat_with_assms.A_interpretation_to_redtriple_orientation[
      OF srm _ t2 E\<^sub>I_wf_carrierS[OF srm ids(2,1)],
      of "pI' n fc" _ _ "map fst fc", OF _ _ _ _ def]

  show ?thesis
  proof ((rule th; (intro allI impI)?), goal_cases)
    case (3 \<alpha>)
    show ?case using check_core_E\<^sub>I[OF srm ids check(1) 3] .
  next
    case (4 \<alpha>)
    show ?case using check_N_for_relative_E\<^sub>I[OF srm ids check(2) 4] .
  qed (insert check_coeffs_E\<^sub>I_final[OF srm ids check_final], auto)
qed


subsection \<open>Connection to Relation_Implementation\<close>

datatype core_matrix_mode = E_I | M_I

type_synonym 'a core_matrix_gt = "showsl + ('a \<Rightarrow> 'a \<Rightarrow> bool)"

definition "core_matrix_gt_valid gtc = (case gtc of Inr gt \<Rightarrow>
   squared_ring_mat_with_assms (\<lambda> x. x \<ge> 1) 1 gt
  | Inl _ \<Rightarrow> True)" 

datatype ('f,'a) core_matrix_inter = Core_Matrix_Inter core_matrix_mode nat "nat list" "('f,'a) fun_coeffs"

lemmas mat_interpretation_code = 
  mat_interpretation_parameter.Matsemiring_def

declare mat_interpretation_code[code]

definition "check_M_I_weak n ids fc = (\<lambda> (l,r). let diff = evalp_rule (Rring n) (pI' n fc) l r
              in check_lpoly_coef_N n diff 
            <+? (\<lambda> e. showsl_lit (STR ''problem in M_I-weak decrease of '') o showsl_rule (l,r) o showsl_nl o 
                 showsl_lit (STR ''with interpretation [left] - [right] = '') o showsl diff o showsl_nl o e))"
definition "check_M_I_strict n gt ids fc = (\<lambda> (l,r). let diff = evalp_rule (Rring n) (pI' n fc) l r in 
             do {check_lpoly_coef_N n diff; check_poly_coef_one_L\<^sub>I n gt ids diff}
            <+? (\<lambda> e. showsl_lit (STR ''problem in M_I-strict decrease of '') o showsl_rule (l,r) o showsl_nl o 
                 showsl_lit (STR ''with interpretation [left] - [right] = '') o showsl diff o showsl_nl o e))"

definition "check_E_I_weak n ids fc = (\<lambda> (l,r). let diff = evalp_rule (Rring n) (pI' n fc) l r in
              check_lpoly_coef_N n diff 
            <+? (\<lambda> e. showsl_lit (STR ''problem in E_I-weak decrease of '') o showsl_rule (l,r) o showsl_nl o
                 showsl_lit (STR ''with interpretation [left] - [right] = '') o showsl diff o showsl_nl o e))"
definition "check_E_I_strict n gt ids fc = (\<lambda> (l,r). let diff = evalp_rule (Rring n) (pI' n fc) l r in 
             do {check_lpoly_coef_N n diff; check_poly_coef_one_P\<^sub>I n gt ids diff}
            <+? (\<lambda> e. showsl_lit (STR ''problem in E_I-strict decrease of '') o showsl_rule (l,r) o showsl_nl o
                 showsl_lit (STR ''with interpretation [left] - [right] = '') o showsl diff o showsl_nl o e))"

 
fun create_core_matrix_rel_impl :: "_ core_matrix_gt \<Rightarrow> ('f :: {showl,compare_order},_)core_matrix_inter \<Rightarrow> ('f,'v :: showl)rel_impl"
  where "create_core_matrix_rel_impl gtc (Core_Matrix_Inter mode n ids fc) = (
     let ns = (case mode of E_I \<Rightarrow> check_E_I_weak n ids fc | 
              M_I \<Rightarrow> check_M_I_weak n ids fc);
         s = (let gt = sum.projr gtc in case mode of E_I \<Rightarrow> check_E_I_strict n gt ids fc | 
              M_I \<Rightarrow> check_M_I_strict n gt ids fc)
          in
    \<lparr>rel_impl.valid = do {gt \<leftarrow> gtc; 
       check_indices n ids; 
       (case mode of 
             E_I \<Rightarrow> check_coeffs_E\<^sub>I_final n ids fc | 
             M_I \<Rightarrow> check_coeffs_M\<^sub>I_final n ids fc)},
     standard = succeed,
     desc = show_core_matrix_inter_main (case mode of E_I \<Rightarrow> STR ''E_I'' | M_I \<Rightarrow> STR ''M_I'') n ids fc,
     s = s,
     ns = ns, 
     nst = ns,
     af = full_af,
     top_af = full_af,
     SN = succeed,
     subst_s = succeed,
     ce_compat = succeed,
     co_rewr = succeed,
     top_mono = succeed,
     top_refl = succeed,
     mono_af = (\<lambda> f. UNIV),
     mono = (\<lambda> _. succeed),
     not_wst = None, 
     not_sst = None, 
     cpx = no_complexity_check \<rparr>)"

lemma create_core_matrix_rel_impl: assumes "core_matrix_gt_valid gtc" 
shows "rel_impl (create_core_matrix_rel_impl gtc mI)"
proof -
  obtain mode n ids fc where mI: "mI = Core_Matrix_Inter mode n ids fc" by (cases mI, auto)
  show ?thesis 
  proof (cases mode)
    case E_I  
    show ?thesis
    unfolding rel_impl_def create_core_matrix_rel_impl.simps mI Let_def rel_impl.simps E_I core_matrix_mode.simps split
    proof (intro allI impI, goal_cases)
      case (1 U)
      from 1 have "isOK(check_indices n ids)" by auto
      from check_indices[OF this] have ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}" by auto
      from 1 have pI_EI: "isOK(check_coeffs_E\<^sub>I_final n ids fc)" by auto
      from 1 obtain gt where gtc: "gtc = Inr gt" and proj: "projr gtc = gt" by (cases gtc, auto)
      from assms[unfolded gtc core_matrix_gt_valid_def]
      have srm: "squared_ring_mat_with_assms ((\<le>) 1) 1 gt"  by auto
      let ?S = "check_E_I_strict n gt ids fc" 
      let ?W = "check_E_I_weak n ids fc" 
      define R where "R = (filter (\<lambda> lr. isOK(?S lr)) U)"
      define S where "S = (filter (\<lambda> lr. isOK(?W lr)) U)"
      have main: "\<exists>s ns. mono_redtriple_order s ns ns \<and> set R \<subseteq> s \<and> set S \<subseteq> ns \<and> ce_compatible s" 
      proof (rule check_E\<^sub>I_to_redtriple_order[OF srm ids])
        have "isOK (check_rules_N_E\<^sub>I n ids fc R)" 
          unfolding check_rules_N_E\<^sub>I_def R_def
          by (auto simp: check_E_I_strict_def check_E_I_weak_def)
        moreover have "isOK (check_rules_one_P\<^sub>I n gt ids fc R)" 
          unfolding check_rules_one_P\<^sub>I_def R_def 
          by (auto simp: check_E_I_strict_def)
        ultimately show "isOK (check_core_E\<^sub>I n gt ids fc R)" using pI_EI
          unfolding check_core_E\<^sub>I_def by auto
        show "isOK (check_N_for_relative_E\<^sub>I n ids fc S)" unfolding check_N_for_relative_E\<^sub>I_def
          using pI_EI by (auto simp: check_rules_N_E\<^sub>I_def S_def check_E_I_weak_def)
      qed 
      then obtain s ns where mro: "mono_redtriple_order s ns ns" and sub: "set R \<subseteq> s" "set S \<subseteq> ns" 
        and ce: "ce_compatible s" 
        by auto
      interpret mono_redtriple_order s ns ns by fact
      from S_imp_NS ce have ce2: "ce_compatible ns" unfolding ce_compatible_def by blast
      show ?case unfolding proj
      proof (rule exI[of _ s], rule exI[of _ ns], rule exI[of _ ns], intro conjI impI allI
          refl_NST ctxt_S ctxt_NS subst_NS subst_S full_af not_subterm_rel_info.simps ce ce2
          compat_S_NS compat_NS_S trans_S trans_NS S_imp_NS SN ctxt_closed_imp_af_monotone top_mono_same)
        show "irrefl s" using SN by (meson SN_on_irrefl irrefl_on_def)
        show "ns \<inter> s\<inverse> = {}" by (simp add: \<open>irrefl s\<close> co_rewrite_irrefl compat_NS_S)
        show "lr \<in> set U \<Longrightarrow> isOK (check_E_I_strict n gt ids fc lr) \<Longrightarrow> lr \<in> s" for lr
          using sub(1) unfolding R_def by auto
        show "lr \<in> set U \<Longrightarrow> isOK (check_E_I_weak n ids fc lr) \<Longrightarrow> lr \<in> ns" for lr
          using sub(2) unfolding S_def by auto
        show "lr \<in> set U \<Longrightarrow> isOK (check_E_I_weak n ids fc lr) \<Longrightarrow> lr \<in> ns" for lr by fact
      qed (auto simp: isOK_no_complexity)
    qed
  next
    case M_I
    show ?thesis 
      unfolding rel_impl_def create_core_matrix_rel_impl.simps mI Let_def rel_impl.simps M_I core_matrix_mode.simps split
    proof (intro allI impI, goal_cases)
      case (1 U)
      from 1 have "isOK(check_indices n ids)" by auto
      from check_indices[OF this] have ids: "set ids \<noteq> {}" "set ids \<subseteq> {0..< n}" by auto
      from 1 have pI_MI: "isOK(check_coeffs_M\<^sub>I_final n ids fc)" by auto
      from 1 obtain gt where gtc: "gtc = Inr gt" and proj: "projr gtc = gt" by (cases gtc, auto)
      from assms[unfolded gtc core_matrix_gt_valid_def]
      have srm: "squared_ring_mat_with_assms ((\<le>) 1) 1 gt"  by auto
      let ?S = "check_M_I_strict n gt ids fc" 
      let ?W = "check_M_I_weak n ids fc" 
      define R where "R = (filter (\<lambda> lr. isOK(?S lr)) U)"
      define S where "S = (filter (\<lambda> lr. isOK(?W lr)) U)"
      have main: "\<exists>s ns. mono_redtriple_order s ns ns \<and> set R \<subseteq> s \<and> set S \<subseteq> ns \<and> ce_compatible s" 
      proof (rule check_M\<^sub>I_to_redtriple_order[OF srm ids])
        have "isOK (check_rules_N_M\<^sub>I n ids fc R)" 
          unfolding check_rules_N_M\<^sub>I_def R_def
          by (auto simp: check_M_I_strict_def check_M_I_weak_def)
        moreover have "isOK (check_rules_one_L\<^sub>I n gt ids fc R)" 
          unfolding check_rules_one_L\<^sub>I_def R_def 
          by (auto simp: check_M_I_strict_def)
        ultimately show "isOK (check_core_M\<^sub>I n gt ids fc R)" using pI_MI 
          unfolding check_core_M\<^sub>I_def by auto
        show "isOK (check_N_for_relative_M\<^sub>I n ids fc S)" unfolding check_N_for_relative_M\<^sub>I_def
          using pI_MI by (auto simp: check_rules_N_M\<^sub>I_def S_def check_M_I_weak_def)
      qed
      then obtain s ns where mro: "mono_redtriple_order s ns ns" and sub: "set R \<subseteq> s" "set S \<subseteq> ns"
        and ce: "ce_compatible s" 
        by auto
      interpret mono_redtriple_order s ns ns by fact
      from S_imp_NS ce have ce2: "ce_compatible ns" unfolding ce_compatible_def by blast
      show ?case unfolding proj
      proof (rule exI[of _ s], rule exI[of _ ns], rule exI[of _ ns], intro conjI impI allI
          refl_NST ctxt_S ctxt_NS subst_NS subst_S full_af not_subterm_rel_info.simps ce ce2
          compat_S_NS compat_NS_S trans_S trans_NS S_imp_NS SN ctxt_closed_imp_af_monotone top_mono_same)
        show "irrefl s" using SN by (meson SN_on_irrefl irrefl_on_def)
        show "ns \<inter> s\<inverse> = {}" by (simp add: \<open>irrefl s\<close> co_rewrite_irrefl compat_NS_S)
        show "lr \<in> set U \<Longrightarrow> isOK (check_M_I_strict n gt ids fc lr) \<Longrightarrow> lr \<in> s" for lr
          using sub(1) unfolding R_def by auto
        show "lr \<in> set U \<Longrightarrow> isOK (check_M_I_weak n ids fc lr) \<Longrightarrow> lr \<in> ns" for lr
          using sub(2) unfolding S_def by auto
        show "lr \<in> set U \<Longrightarrow> isOK (check_M_I_weak n ids fc lr) \<Longrightarrow> lr \<in> ns" for lr by fact
      qed (auto simp: isOK_no_complexity)
    qed
  qed
qed

declare create_core_matrix_rel_impl.simps[simp del]

definition core_matrix_gt_int :: "int core_matrix_gt" where 
  "core_matrix_gt_int = Inr (\<lambda> x y. x > y)" 

lemma core_matrix_gt_int: "core_matrix_gt_valid core_matrix_gt_int" 
  unfolding core_matrix_gt_valid_def core_matrix_gt_int_def
  using squared_int_mat by auto

definition core_matrix_gt_delta :: "'a :: {floor_ceiling,showl} \<Rightarrow> 'a core_matrix_gt" where 
  "core_matrix_gt_delta d = (do {
     check (0 < d \<and> d \<le> 1) (showsl_lit (STR ''parameter delta '') o showsl d
       o showsl_lit (STR '' must be between 0 and 1''));
     return (delta_gt d)})" 

lemma core_matrix_gt_delta: "core_matrix_gt_valid (core_matrix_gt_delta d)" 
  unfolding core_matrix_gt_valid_def core_matrix_gt_delta_def
  using squared_ring_mat_delta[unfolded delta_mono_def] 
  by (cases "0 < d \<and> d \<le> 1", auto simp: check_def)


hide_const Ip I

definition create_core_matrix_int where 
  "create_core_matrix_int rel = (case rel of Core_Matrix_Inter E_I n I fk_cs
      \<Rightarrow> core_EI_rel_impl_int n I fk_cs
   | _ \<Rightarrow> create_core_matrix_rel_impl core_matrix_gt_int rel)" 

lemma create_core_matrix_int: "rel_impl (create_core_matrix_int rel)" 
proof (cases rel)
  case rel: (Core_Matrix_Inter mode n I fk_cs)
  show ?thesis
  proof (cases mode)
    case E_I
    show ?thesis unfolding rel E_I create_core_matrix_int_def 
      by (auto simp: core_EI_rel_impl_int)
  next
    case M_I
    show ?thesis unfolding rel M_I create_core_matrix_int_def
      using create_core_matrix_rel_impl[OF core_matrix_gt_int]
      by auto
  qed
qed

definition create_core_matrix_fract where 
  "create_core_matrix_fract delta rel = (case rel of Core_Matrix_Inter E_I n I fk_cs
      \<Rightarrow> core_EI_rel_impl_fract n delta I fk_cs
   | _ \<Rightarrow> create_core_matrix_rel_impl (core_matrix_gt_delta delta) rel)" 

lemma create_core_matrix_fract: "rel_impl (create_core_matrix_fract delta rel)" 
proof (cases rel)
  case rel: (Core_Matrix_Inter mode n I fk_cs)
  show ?thesis
  proof (cases mode)
    case E_I
    show ?thesis unfolding rel E_I create_core_matrix_fract_def 
      by (auto simp: core_EI_rel_impl_fract)
  next
    case M_I
    show ?thesis unfolding rel M_I create_core_matrix_fract_def
      using create_core_matrix_rel_impl[OF core_matrix_gt_delta]
      by auto
  qed
qed

end