{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE ScopedTypeVariables #-}

import Control.Monad (
  unless,
 )
import Data.LCTRS.FIdentifier (FId (..), getFIdSort)
import qualified Data.LCTRS.Guard as G
import Data.LCTRS.LCTRS (
  LCTRS,
  checkLhsRules,
  getDefines,
  getFuns,
  getRules,
  getTheories,
 )
import qualified Data.LCTRS.Rule as R
import Data.LCTRS.Sort (SortAnnotation, inSort, outSort)
import Data.LCTRS.Term (Term (Fun, Var))
import Data.LCTRS.VIdentifier (VId)
import Data.Monad
import Data.SMT (Theory (FixedSizeBitVectors, Reals, Reals_Ints))
import qualified Data.Set as S
import Options.Applicative hiding (
  Failure,
  Success,
 )
import Parser.Ari (getLogic)
import qualified Parser.Ari as Ari
import Prettyprinter (Doc, Pretty (..), enclose, encloseSep, line, vsep, (<+>))
import qualified Prettyprinter as PP
import Type.SortLCTRS (DefaultFId, DefaultVId)
import Type.TypeChecking (typeCheck)
import Type.TypeInference (deriveTypes)

data Args = Args
  { filepath :: FilePath
  }

main :: IO ()
main = do
  args <- execParser opts
  (inputLCTRS, _) <- execFreshM 0 $ parseAndCheckLCTRS args
  -- print $ prettyLCTRSDefault inputLCTRS
  print $ prettyCoraLCTRS inputLCTRS
 where
  opts =
    info
      (helper <*> arguments)
      ( fullDesc
          PP.<> progDesc
            "This tool tries to transform (if possible) an LCTRS in ARI format to the format of Cora.\
            \ Due to various reasons it could be still the case that Cora does not accept the result."
          PP.<> header "ARI input file to Cora input file."
      )

prettyCoraLCTRS
  :: forall v ann. (Ord v, Pretty v) => LCTRS (FId DefaultFId) (VId v) -> Doc ann
prettyCoraLCTRS lctrs | not $ null $ getDefines lctrs = error "Cora cannot specify custom theory symbols."
prettyCoraLCTRS lctrs | Reals `elem` getTheories lctrs = error "Cora does not support with Reals."
prettyCoraLCTRS lctrs | Reals_Ints `elem` getTheories lctrs = error "Cora does not support with Reals_Ints."
prettyCoraLCTRS lctrs
  | FixedSizeBitVectors `elem` getTheories lctrs =
      error "Cora does not support with FixedSizeBitVectors."
prettyCoraLCTRS lctrs =
  vsep
    [ prettySignature
    , line
    , prettyRules
    ]
 where
  prettySignature =
    let fs = getFuns lctrs
    in  vsep $ map (\f -> pretty f <> " :: " <> prettySortAnn (getFIdSort f)) $ S.toList fs

  prettySortAnn :: SortAnnotation -> Doc ann
  prettySortAnn sortAnnotation
    | null (inSort sortAnnotation) = pretty $ outSort sortAnnotation
    | length (inSort sortAnnotation) == 1 =
        pretty (head $ inSort sortAnnotation)
          <+> "->"
          <+> pretty (outSort sortAnnotation)
    | otherwise =
        encloseSep "" "" " -> " (map pretty $ inSort sortAnnotation)
          <+> "->"
          <+> pretty (outSort sortAnnotation)

  prettyRules = vsep $ map prettyCoraRule $ getRules lctrs

prettyCoraRule :: (Pretty v, Ord v) => R.Rule (FId DefaultFId) v -> Doc ann
prettyCoraRule rule =
  prettyCoraTerm (R.lhs rule)
    <+> "->"
    <+> prettyCoraTerm (R.rhs rule)
    <+> "|"
    <+> prettyCoraConstraint (R.guard rule)

prettyCoraConstraint :: (Eq v, Pretty v) => G.Guard (FId DefaultFId) v -> Doc ann
prettyCoraConstraint = prettyCoraTerm . G.collapseGuardToTerm

prettyCoraTerm :: (Pretty v) => Term (FId DefaultFId) v -> Doc ann
prettyCoraTerm (Var v) = pretty v
prettyCoraTerm (Fun _ f []) = prettyF f
prettyCoraTerm (Fun _ f@Conj args) = prettyInfix f args
prettyCoraTerm (Fun _ f@Disj args) = prettyInfix f args
prettyCoraTerm (Fun _ Imp [a, b]) = prettyF Neg <> enclose "(" ")" (prettyCoraTerm a <+> prettyF Disj <+> prettyCoraTerm b)
prettyCoraTerm (Fun _ f@(Eq _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(GrT _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "<" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId ">=" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "<=" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "+" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "-" _) args) | length args > 1 = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "*" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ f@(FId "/" _) args) = prettyInfix f args
prettyCoraTerm (Fun _ (FId "mod" sa) args) = prettyInfix (FId "%" sa) args
prettyCoraTerm (Fun _ f args) = prettyF f <> encloseSep "(" ")" ", " (map prettyCoraTerm args)

prettyInfix :: (Pretty v) => FId DefaultFId -> [Term (FId DefaultFId) v] -> Doc ann
prettyInfix f args = encloseSep "" "" (enclose " " " " $ prettyF f) (map prettyCoraTerm args)

prettyF :: () => FId DefaultFId -> Doc ann
prettyF Conj = "∧"
prettyF Disj = "∨"
prettyF Neg = "¬"
prettyF f = pretty f

parseAndCheckLCTRS :: Args -> FreshM (LCTRS (FId DefaultFId) (VId DefaultVId), Logic)
parseAndCheckLCTRS args = do
  let file = filepath args
  input <- Ari.fromFile' file
  let lctrs = deriveTypes input
  unless
    (typeCheck (Ari.getFSorts input) (Ari.getThSorts input) lctrs)
    (error "Type checking failed...")
  unless (checkLhsRules lctrs) $
    error
      "Left-hand sides of rules must have a term symbol at root position."
  return (lctrs, getLogic input)

arguments :: Parser Args
arguments =
  Args
    <$> argument
      str
      (metavar "FILEPATH" PP.<> help "Path to the source file")
