{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE PatternSynonyms #-}
{-# LANGUAGE ScopedTypeVariables #-}

import Control.Monad (
  unless,
 )
import Data.Containers.ListUtils (nubOrd)
import Data.LCTRS.FIdentifier (FId, getFIdSort)
import qualified Data.LCTRS.Guard as G
import Data.LCTRS.LCTRS (
  LCTRS,
  checkLhsRules,
  getDefines,
  getFuns,
  getRules,
  getTheories,
 )
import qualified Data.LCTRS.Rule as R
import Data.LCTRS.Sort (Sorted (sort), inSort, outSort)
import qualified Data.LCTRS.Term as T
import Data.LCTRS.VIdentifier (VId)
import Data.Monad
import Data.SMT (Theory (..))
import qualified Data.Set as S
import Options.Applicative hiding (
  Failure,
  Success,
 )
import Parser.Ari (getLogic)
import qualified Parser.Ari as Ari
import Prettyprinter
import qualified Prettyprinter as PP
import Type.SortLCTRS (DefaultFId, DefaultVId)
import Type.TypeChecking (typeCheck)
import Type.TypeInference (deriveTypes)

data Args = Args
  { filepath :: FilePath
  }

main :: IO ()
main = do
  args <- execParser opts
  (inputLCTRS, _) <- execFreshM 0 $ parseAndCheckLCTRS args
  -- print $ prettyLCTRSDefault inputLCTRS
  print $ prettyCrestCadeLCTRS inputLCTRS
 where
  opts =
    info
      (helper <*> arguments)
      ( fullDesc
          PP.<> progDesc
            "This tool tries to transform (if possible) an LCTRS in ARI format to the format of crest submitted for CADE.\
            \ Due to various reasons it could be still the case that crest does not accept the result."
          PP.<> header "ARI input file to crest (CADE version) input file."
      )

prettyCrestCadeLCTRS :: (Ord v, Eq f, Pretty f, Pretty v) => LCTRS (FId f) (VId v) -> Doc ann
prettyCrestCadeLCTRS lctrs
  | not $ null $ getDefines lctrs = error "crest (CADE version) cannot specify custom theory symbols."
prettyCrestCadeLCTRS lctrs =
  vsep $
    ("(format LCTRS" <> prettyLogic <> ")")
      : map pFun (S.toList $ getFuns lctrs)
      ++ map pRule (getRules lctrs)
 where
  pFun f =
    let sa = getFIdSort f
    in  parens $
          "fun"
            <+> pretty f
            <+> pretty (length $ inSort sa)
            <+> pSortAnn (inSort sa, outSort sa)

  pSortAnn ([], s) = ":sort" <+> parens (pretty s)
  pSortAnn (ss, s) =
    ":sort" <+> parens (hsep (map pretty ss) <+> pretty s)

  pRule rule =
    parens $
      "rule"
        <+> pTerm (R.lhs rule)
        <+> pTerm (R.rhs rule)
        <+> ":guard"
        <+> pTerm (G.collapseGuardToTerm $ R.guard rule)
        <+> ":vars"
        <+> pVarAnn rule

  pVarAnn rule =
    let vs = nubOrd (R.vars rule)
    in  parens (hsep $ map (\v -> parens $ pretty v <+> pretty (sort v)) vs)

  pTerm (T.Var v) = pretty v
  pTerm (T.Fun _ f []) = pretty f
  pTerm (T.Fun _ f as) = parens $ pretty f <+> hsep (map pTerm as)

  prettyLogic = go (getTheories lctrs)
   where
    go [] = ""
    go [Core] = ""
    go [Ints] = " :logic QF_NIA"
    go [Reals] = " :logic QF_LRA"
    go [_] = error "crest (CADE version) accepts the given theory most likely not."
    go (Core : ts) = go ts
    go (th : [Core]) = go [th]
    go (_ : _) = error "crest (CADE version) accepts only one theory."

parseAndCheckLCTRS :: Args -> FreshM (LCTRS (FId DefaultFId) (VId DefaultVId), Logic)
parseAndCheckLCTRS args = do
  let file = filepath args
  input <- Ari.fromFile' file
  let lctrs = deriveTypes input
  unless
    (typeCheck (Ari.getFSorts input) (Ari.getThSorts input) lctrs)
    (error "Type checking failed...")
  unless (checkLhsRules lctrs) $
    error
      "Left-hand sides of rules must have a term symbol at root position."
  return (lctrs, getLogic input)

arguments :: Parser Args
arguments =
  Args
    <$> argument
      str
      (metavar "FILEPATH" PP.<> help "Path to the source file")
