{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE PatternSynonyms #-}
{-# LANGUAGE ScopedTypeVariables #-}

import Control.Monad (
  unless,
 )
import Data.LCTRS.FIdentifier (FId (Conj, Disj), getFIdInSort)
import qualified Data.LCTRS.Guard as G
import Data.LCTRS.LCTRS (
  LCTRS,
  allFunsInRules,
  checkLhsRules,
  getDefines,
  getFuns,
  getRules,
  getTheories,
 )
import Data.LCTRS.Rule (lhs)
import qualified Data.LCTRS.Rule as R
import Data.LCTRS.Sort (Attr (AttrInt), Sort (AttrSort), Sorted (sort))
import Data.LCTRS.Term (FunType (TheorySym), Term (Fun, Var), pattern Val)
import qualified Data.LCTRS.Term as T
import Data.LCTRS.VIdentifier (VId)
import Data.Monad
import Data.SMT (Theory (..), isBVSort)
import qualified Data.Set as S
import Options.Applicative hiding (
  Failure,
  Success,
 )
import Parser.Ari (getLogic)
import qualified Parser.Ari as Ari
import Prettyprinter (Doc, Pretty (..), encloseSep, line, vsep, (<+>))
import qualified Prettyprinter as PP
import Type.SortLCTRS (DefaultFId, DefaultVId)
import Type.TypeChecking (typeCheck)
import Type.TypeInference (deriveTypes)

data Args = Args
  { filepath :: FilePath
  , sn :: Bool
  , cr :: Bool
  , custom :: Maybe String
  }

main :: IO ()
main = do
  args <- execParser opts
  (inputLCTRS, _) <- execFreshM 0 $ parseAndCheckLCTRS args
  -- print $ prettyLCTRSDefault inputLCTRS
  print $ prettyCtrlLCTRS args inputLCTRS
 where
  opts =
    info
      (helper <*> arguments)
      ( fullDesc
          PP.<> progDesc
            "This tool tries to transform (if possible) an LCTRS in ARI format to the format of Ctrl.\
            \ Due to various reasons it could be still the case that Ctrl does not accept the result."
          PP.<> header "ARI input file to Ctrl input file."
      )

prettyCtrlLCTRS
  :: forall f v ann. (Ord v, Ord f, Pretty f, Pretty v) => Args -> LCTRS (FId f) (VId v) -> Doc ann
prettyCtrlLCTRS _ lctrs | not $ null $ getDefines lctrs = error "Ctrl cannot specify custom theory symbols."
prettyCtrlLCTRS args lctrs =
  vsep
    [ prettyTheory
    , prettyLogic
    , prettySolver
    , prettySignature
    , prettyRules
    , prettyNonStandard
    , prettyIrregular
    , prettyQuery
    ]
 where
  bitwidth FixedSizeBitVectors =
    let
      bvSorts = concatMap (\f -> filter isBVSort $ sort f : getFIdInSort f) $ allFunsInRules lctrs
      extract (AttrSort "BitVec" (AttrInt b)) = b
      extract _ = error "unknown bit-width for symbol; Ctrl cannot anlyze this."
    in
      case bvSorts of
        [] -> 1 -- NOTE: make Ctrl just analyze file
        (bV : bVs) | all ((== extract bV) . extract) bVs -> extract bV
        _ -> error "different bit-widths; Ctrl cannot anlyze this."
  bitwidth _ = error "bit width does only make sense for bitvectors."
  sep = ";"
  prettyTheory = go (getTheories lctrs) <> sep
   where
    go [] = "THEORY"
    go [Core] = "THEORY core"
    go [Ints] = "THEORY ints"
    go [FixedSizeBitVectors] = "THEORY bitvectors" <> pretty (bitwidth FixedSizeBitVectors)
    go [_] = error "Ctrl accepts the given theory most likely not."
    go (Core : ts) = go ts
    go (th : [Core]) = go [th]
    go (_ : _) = error "Ctrl accepts only one theory."
  prettyLogic = go (getTheories lctrs) <> sep
   where
    go [] = "LOGIC"
    go [Core] = "LOGIC QF_UF"
    go [Ints] = "LOGIC NIA"
    go [FixedSizeBitVectors] = "LOGIC QF_BV"
    go [_] = error "Ctrl accepts the given theory most likely not."
    go (Core : ts) = go ts
    go (th : [Core]) = go [th]
    go (_ : _) = error "Ctrl accepts only one theory."
  prettySolver = "SOLVER internal" <> sep
  prettySignature =
    let fs = getFuns lctrs
    in  "SIGNATURE"
          <+> foldr (\f acc -> pretty f <> ", " <> acc) "" (S.toList fs)
          <> go (getTheories lctrs)
          <> sep
   where
    go [] = ""
    go [Core] = ""
    go [Ints] = "!INTEGER"
    go [FixedSizeBitVectors] = "" -- TODO: correct?
    go [_] = error "Ctrl accepts the given theory most likely not."
    go (Core : ts) = go ts
    go (th : [Core]) = go [th]
    go (_ : _) = error "Ctrl accepts only one theory."

  prettyRules = "RULES" <> line <> vsep (map ((<> sep) . prettyCtrlRule) (getRules lctrs))

  prettyNonStandard
    | any isNonStandard (getRules lctrs) = "NON-STANDARD"
    | otherwise = ""

  prettyIrregular
    | any isIrregular (getRules lctrs) = "IRREGULAR"
    | otherwise = ""

  prettyQuery
    | sn args = "QUERY termination"
    | cr args = "QUERY confluence"
    | otherwise =
        case custom args of
          Just query -> "QUERY " <> pretty query
          Nothing -> "QUERY termination"

prettyCtrlRule :: (Pretty v, Pretty f, Eq f, Ord v) => R.Rule (FId f) v -> Doc ann
prettyCtrlRule rule =
  prettyCtrlTerm (const False) (R.lhs rule)
    <+> "->"
    <+> prettyCtrlTerm isRandomVar (R.rhs rule)
    <+> "["
    <> prettyCtrlConstraint (R.guard rule)
    <> "]"
 where
  isRandomVar v = v `S.member` R.extraVars rule

prettyCtrlConstraint :: (Eq v, Eq f, Pretty v, Pretty f) => G.Guard (FId f) v -> Doc ann
prettyCtrlConstraint = prettyCtrlTerm (const False) . G.collapseGuardToTerm

prettyCtrlTerm :: (Pretty v, Pretty f) => (v -> Bool) -> Term (FId f) v -> Doc ann
prettyCtrlTerm isRandomVar (Var v) | isRandomVar v = "inp_" <> pretty v
prettyCtrlTerm _ (Var v) = pretty v
prettyCtrlTerm _ (Fun _ f []) = prettyF f
prettyCtrlTerm isRandomVar (Fun _ f args) = prettyF f <> encloseSep "(" ")" ", " (map (prettyCtrlTerm isRandomVar) args)

prettyF :: (Pretty f) => FId f -> Doc ann
prettyF Conj = "/\\"
prettyF Disj = "\\/"
prettyF f = pretty f

isIrregular :: (Ord v) => R.Rule f v -> Bool
isIrregular rule = any (\v -> v `S.notMember` S.fromList (T.vars $ R.lhs rule)) (G.varsGuard $ R.guard rule)

isNonStandard :: R.Rule f v -> Bool
isNonStandard = isNonValThSym . lhs
 where
  isNonValThSym (Var _) = False
  isNonValThSym (Val _) = False
  isNonValThSym (Fun TheorySym _ _) = True
  isNonValThSym (Fun _ _ args) = any isNonValThSym args

parseAndCheckLCTRS :: Args -> FreshM (LCTRS (FId DefaultFId) (VId DefaultVId), Logic)
parseAndCheckLCTRS args = do
  let file = filepath args
  input <- Ari.fromFile' file
  let lctrs = deriveTypes input
  unless
    (typeCheck (Ari.getFSorts input) (Ari.getThSorts input) lctrs)
    (error "Type checking failed...")
  unless (checkLhsRules lctrs) $
    error
      "Left-hand sides of rules must have a term symbol at root position."
  return (lctrs, getLogic input)

arguments :: Parser Args
arguments =
  Args
    <$> argument
      str
      (metavar "FILEPATH" PP.<> help "Path to the source file")
    <*> switch
      (long "sn" PP.<> help "Termination Analysis (default).")
    <*> switch
      (long "cr" PP.<> help "Confluence Analysis.")
    <*> optional
      ( option
          str
          ( metavar "STRING"
              PP.<> long "custom"
              PP.<> short 'c'
              PP.<> help "Custom QUERY for Ctrl."
          )
      )
