import Data.Monad (FreshM, execFreshM)
import qualified Data.SMT as SMT
import Data.Text (Text)
import Data.Void (Void)
import Options.Applicative
import qualified Parser.Ari as Ari
import Text.Megaparsec

--------------------------------------------------------------------------------
-- types
--------------------------------------------------------------------------------

data Args = Args
  { tag :: InputTag
  , filepath :: FilePath
  }

data Answer = MAYBE | NO | YES
  deriving (Show)

--------------------------------------------------------------------------------
-- tags
--------------------------------------------------------------------------------

data InputTag = Lctrs
  deriving (Eq, Ord)

-- NOTE add new tags here
fromString :: String -> Maybe InputTag
fromString "lctrs" = Just Lctrs
fromString _ = Nothing

--------------------------------------------------------------------------------
-- main functionality
--------------------------------------------------------------------------------

main :: IO ()
main = do
  args <- execParser opts
  format <- do
    format <- execFreshM 0 $ parseFormat (filepath args)
    case format of
      (Left e, _) -> error $ errorBundlePretty e
      (Right f, _) -> return f
  print $ answer format (tag args)

answer :: Ari.Format -> InputTag -> Answer
answer Ari.F_LCTRS Lctrs = YES
answer _ Lctrs = NO

--------------------------------------------------------------------------------
-- input tag parser
--------------------------------------------------------------------------------

parseTag :: String -> Either String InputTag
parseTag s =
  case parse (go <* eof) "" s :: Either (ParseErrorBundle String Void) InputTag of
    Left e -> Left $ errorBundlePretty e
    Right t -> Right t
 where
  go = do
    iden <- takeWhile1P Nothing (const True) -- parse everything
    case fromString iden of -- check that iden is valid tag
      Nothing ->
        fail $ "unknown tag \"" ++ iden ++ "\" - please consult help text for more information"
      Just t -> return t

--------------------------------------------------------------------------------
-- format parser
--------------------------------------------------------------------------------

parseFormat :: FilePath -> FreshM (Either (ParseErrorBundle Text Void) Ari.Format, Ari.DefaultInput)
parseFormat s = Ari.fromFileFun s (SMT.whitespace *> Ari.pFormat)

--------------------------------------------------------------------------------
-- input handling
--------------------------------------------------------------------------------

opts :: ParserInfo Args
opts =
  info
    (helper <*> arguments)
    ( fullDesc
        <> progDesc "Checks that an LCTRS fulfills the conditions for a tag."
        <> header "Checks that an LCTRS fulfills the conditions for a tag."
    )

arguments :: Parser Args
arguments =
  Args
    <$> argument
      (eitherReader parseTag)
      (metavar "TAG" <> help "one of [lctrs]")
    <*> argument str (metavar "FILEPATH" <> help "Path to the source file")
