{- |
Module      : CheckingConfluence
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the main functionality to check a given LCTRS for the confluence
property.
-}
module Analysis.Confluence.CheckingConfluence (
  -- * Confluence Analysis
  analyzeConfluence,
  analyzeConfluenceSeq,
  defaultStrategy,

  -- * Wrapper Functions for (Non-)CR Methods
  collectMethods,

  -- * Helper Functions
  handleError,
  isCR,
  isNonCR,
  analyzedCR,
)
where

import Analysis.Confluence.Confluence (CRResult (..))
import Analysis.Confluence.DevelopmentClosedness (
  isAlmostDevelopmentClosed,
  isAlmostDevelopmentClosedH,
 )
import Analysis.Confluence.NewmansLemma (isCRbyNewmansLemma)
import Analysis.Confluence.NonConfluence (checkNonConfluent)
import Analysis.Confluence.StrongClosedness (isStrongClosed)
import Analysis.Confluence.Toyama81 (toyama81Applies)
import Analysis.Confluence.Transformation.RedundantRules (
  addCPJoiningRules,
  addRewrittenRhss,
  crByRedundantRules,
  removeJoinableRules,
 )
import Control.Monad.IO.Class (MonadIO (liftIO))
import Data.LCTRS.FIdentifier (
  FId,
 )
import Data.LCTRS.LCTRS (
  LCTRS,
  definesToSExpr,
 )
import Data.LCTRS.VIdentifier (
  VId,
 )
import Data.Maybe (fromMaybe)
import Data.Monad
import Data.SExpr
import qualified Data.Set as S
import Parser.Strategy (CRProcessor)
import Prettyprinter
import Rewriting.CriticalPair (CriticalPair, computeCPs)
import Rewriting.ParallelCriticalPair (computeParallelCPs)
import SimpleSMT (Solver (stop))
import Utils

----------------------------------------------------------------------------------------------------
-- confluence analysis
----------------------------------------------------------------------------------------------------

-- | isolated compuation in the StateM monad
evalIsolatedComputation
  :: (Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => Int
  -> Bool
  -> ExtSolver
  -> LCTRS f v
  -> StateM a
  -> FreshM (Either Error a)
evalIsolatedComputation solverTO debugFlag extsolver lctrs computation = do
  s <- liftIO $ chooseSolver solverTO debugFlag extsolver
  val <- evalComp s (definesToSExpr lctrs) False computation
  val <$ liftIO (stop s)

-- | confluence analysis with all methods executed asynchronously
analyzeConfluence
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => ExtSolver
  -> [Solver]
  -> Maybe [CRProcessor (FId f) (VId v)]
  -> Bool
  -> Int
  -> LCTRS (FId f) (VId v)
  -> FreshM CRResult
analyzeConfluence extsolver solvers methodI debugFlag solverTO lctrs = do
  computationResult <- evalIsolatedComputation solverTO debugFlag extsolver lctrs $ do
    computeCPs lctrs
  case computationResult of
    Left error ->
      return $ handleError debugFlag $ Left error
    Right cps -> do
      let collectedMethods = collectMethods lctrs cps methodI
      let matchSolverMethod = lenSensitiveZip collectedMethods solvers
      next <- freshI
      let computations =
            [ evalStateM solver (definesToSExpr lctrs) debugFlag next method
            | (method, solver) <- matchSolverMethod
            ]
      jobs <- liftIO $ asyncs computations
      result <- liftIO $ waitF analyzedCR (Right MaybeConfluent) $ S.fromList jobs
      return $ handleError debugFlag result

-- | confluence analysis with all methods executed sequentially
analyzeConfluenceSeq
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => ExtSolver
  -> [Solver]
  -> Maybe [CRProcessor (FId f) (VId v)]
  -> Bool
  -> Int
  -> LCTRS (FId f) (VId v)
  -> FreshM CRResult
analyzeConfluenceSeq extsolver solvers methodI debugFlag solverTO lc = do
  computationResult <- evalIsolatedComputation solverTO debugFlag extsolver lc $ do
    computeCPs lc
  case computationResult of
    Left error ->
      return $ handleError debugFlag $ Left error
    Right cps -> do
      let collectedMethods = collectMethods lc cps methodI
      let matchSolverMethod = lenSensitiveZip collectedMethods solvers
      next <- freshI
      let computations =
            [ evalStateM solver (definesToSExpr lc) debugFlag next method
            | (method, solver) <- matchSolverMethod
            ]
      wasFound <- liftIO $ findMs analyzedCR computations
      return . handleError debugFlag $
        fromMaybe (Right MaybeConfluent) wasFound

----------------------------------------------------------------------------------------------------
-- methods
----------------------------------------------------------------------------------------------------

-- | collect methods from the argument integer list
collectMethods
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> Maybe [CRProcessor (FId f) (VId v)]
  -> [StateM CRResult]
-- collectMethods lc cps = maybe (applyArgs $ map snd3 processorsCrNcr) applyArgs
--  where
--   applyArgs = map (($ cps) . ($ lc))
collectMethods lc cps = maybe (defaultStrategy lc cps) applyArgs
 where
  applyArgs = map (($ cps) . ($ lc))

defaultStrategy
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> [StateM CRResult]
defaultStrategy lc cps =
  map
    (($ cps) . ($ lc))
    [ isStrongClosed
    , isAlmostDevelopmentClosed
    , isAlmostDevelopmentClosedH
    , checkNonConfluent
    , \l cs -> computeParallelCPs l >>= toyama81Applies l cs
    , isCRbyNewmansLemma
    , crByRedundantRules addCPJoiningRules isAlmostDevelopmentClosed
    , crByRedundantRules addRewrittenRhss isAlmostDevelopmentClosed
    , crByRedundantRules removeJoinableRules isAlmostDevelopmentClosed
    ]

----------------------------------------------------------------------------------------------------
-- helper functions
----------------------------------------------------------------------------------------------------

-- | handle an error returned from a computation
handleError :: Bool -> Either Error CRResult -> CRResult
handleError False (Left (MayError _)) = MaybeConfluent
-- for debugging if the result is MAYBE because of an non severe error
handleError True (Left (MayError s)) = error $ "A non-severe error occured in the confluence analysis " ++ s
handleError False (Left (SevError _)) = MaybeConfluent
-- for debugging if the result is MAYBE because of an error
handleError True (Left (SevError s)) = error $ "A severe error occured in the confluence analysis " ++ s
handleError _ (Right r) = r

-- | does the given Either type consists of a confluence result
isCR :: Either Error CRResult -> Bool
isCR (Right (Confluent _)) = True
-- for debugging if the result is MAYBE because of an non severe error
-- isCR (Left (MayError e)) = error $ "A non-severe error occured in the confluence analysis " ++ e -- Debugging
isCR (Left (SevError e)) = error $ "A severe error occured in the confluence analysis " ++ e -- Debugging
isCR _ = False

-- | does the given Either type consists of a non-confluence result
isNonCR :: Either Error CRResult -> Bool
isNonCR (Right (NonConfluent _)) = True
-- for debugging if the result is MAYBE because of an non severe error
-- isNonCR (Left (MayError e)) = error $ "A non-severe error occured in the confluence analysis " ++ e -- Debugging
isNonCR (Left (SevError e)) = error $ "A severe error occured in the confluence analysis " ++ e -- Debugging
isNonCR _ = False

-- | does the given Either type consists of a (non-)confluence result
analyzedCR :: Either Error CRResult -> Bool
analyzedCR (Right (Confluent _)) = True
analyzedCR (Right (NonConfluent _)) = True
-- for debugging if the result is MAYBE because of an non severe error
-- analyzedCR (Left (MayError e)) = error $ "A non-severe error occured in the confluence analysis " ++ e -- Debugging
analyzedCR (Left (SevError e)) = error $ "A severe error occured in the confluence analysis " ++ e -- Debugging
analyzedCR _ = False
