{- |
Module      : Confluence
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the main types and auxiliary function in order to perform a confluence analysis.
-}
module Analysis.Confluence.Confluence where

import Data.LCTRS.FIdentifier (
  FId,
 )
import Data.LCTRS.Guard (
  conjGuards,
  varsGuard,
 )
import qualified Data.LCTRS.Guard as G
import Data.LCTRS.LCTRS (
  CRSeq (..),
  StepAnn,
  combineCRSeqs,
 )
import Data.LCTRS.Sort (Sorted)
import qualified Data.LCTRS.Term as T
import Data.Monad (StateM)
import Data.SExpr (ToSExpr)
import Prettyprinter
import Rewriting.ConstrainedRewriting.ConstrainedRewriting (
  equivalentUnderConstraint,
  trivialConstrainedEq,
 )
import Utils (findMs)

data CRResult = Confluent (CRMethod, String) | NonConfluent (NCRMethod, String) | MaybeConfluent
  deriving (Eq)

data CRMethod
  = Orthogonality
  | WeakOrthogonality
  | StrongClosedness
  | ParallelClosedness
  | AlmostParallelClosedness
  | DevelopmentClosedness
  | AlmostDevelopmentClosedness
  | NewmansLemma
  | Toyama81
  deriving (Eq)

prettyCRMethod :: CRMethod -> Doc ann
prettyCRMethod Orthogonality = "Orthogonality"
prettyCRMethod WeakOrthogonality = "Weak Orthogonality"
prettyCRMethod StrongClosedness = "Strong Closedness"
prettyCRMethod ParallelClosedness = "Parallel Closedness"
prettyCRMethod AlmostParallelClosedness = "Almost Parallel Closedness"
prettyCRMethod DevelopmentClosedness = "Development Closedness"
prettyCRMethod AlmostDevelopmentClosedness = "Almost Development Closedness"
prettyCRMethod NewmansLemma = "NewmansLemma"
prettyCRMethod Toyama81 = "Toyama81"

data NCRMethod = TwoDifferentNFs
  deriving (Eq)

prettyNCRMethod :: NCRMethod -> Doc ann
prettyNCRMethod TwoDifferentNFs = "\"two different NFs found\""

data Answer = Maybe | No | Yes
  deriving (Eq)

prettyAnswer :: Answer -> Doc ann
prettyAnswer Maybe = "MAYBE"
prettyAnswer Yes = "YES"
prettyAnswer No = "NO"

prettyCRResult :: CRResult -> Doc ann
prettyCRResult (Confluent (s, p)) =
  "Confluent by"
    <+> prettyCRMethod s
    <+> "with proof:"
    <> line
    <> indent 2 (pretty p)
prettyCRResult (NonConfluent (s, p)) =
  "Not confluent by"
    <+> prettyNCRMethod s
    <+> "with proof:"
    <> line
    <> indent 2 (pretty p)
prettyCRResult MaybeConfluent = "Confluence could not be determined."

crBoolToResult :: (CRMethod, String) -> Bool -> CRResult
crBoolToResult p True = Confluent p
crBoolToResult _ False = MaybeConfluent

ncrBoolToResult :: (NCRMethod, String) -> Bool -> CRResult
ncrBoolToResult p True = NonConfluent p
ncrBoolToResult _ False = MaybeConfluent

crresultToAnswer :: CRResult -> Answer
crresultToAnswer (Confluent _) = Yes
crresultToAnswer (NonConfluent _) = No
crresultToAnswer MaybeConfluent = Maybe

crresultToMethod :: CRResult -> Maybe (Either CRMethod NCRMethod)
crresultToMethod (Confluent (m, _)) = Just $ Left m
crresultToMethod (NonConfluent (m, _)) = Just $ Right m
crresultToMethod MaybeConfluent = Nothing

showCRresultMethod :: CRResult -> String
showCRresultMethod (Confluent (m, _)) = show $ prettyCRMethod m
showCRresultMethod (NonConfluent (m, _)) = show $ prettyNCRMethod m
showCRresultMethod MaybeConfluent = "---"

-- helper functions for confluence methods

findEquivs
  :: (Ord v, Ord f, ToSExpr v, ToSExpr f, Pretty v, Pretty f, Sorted v)
  => [CRSeq (FId f) v]
  -> [CRSeq (FId f) v]
  -> StateM (Maybe (CRSeq (FId f) v, CRSeq (FId f) v))
findEquivs seqsS seqsT = do
  -- let equivs =
  --       [ ((CRSeq seqS, CRSeq seqT), ) <$> equivalentUnderConstraint s
  --                                                        t
  --                                                        phis
  --                                                        (acceptable phis)
  --                                                        (getValueCheck lc)
  --       | (CRSeq seqS) <- seqsS
  --       , (CRSeq seqT) <- seqsT
  --       , not (null seqS)
  --       , not (null seqT)
  --       , let ((s,g1):_) = seqS
  --       , let ((t,g2):_) = seqT
  --       , let phis = g1 ++ g2
  --       ]
  -- let equivs = do
  --         let poss = do
  --               (CRSeq seqS) <- seqsS
  --               (CRSeq seqT) <- seqsT
  --               case (seqS,seqT) of
  --                 ([],_) -> []
  --                 (_,[]) -> []
  --                 ((s,g1):_,(t,g2):_) -> do
  --                   let phis = conjGuards g1 g2
  --                   return ((CRSeq seqS, CRSeq seqT), (s,t,phis))
  --         map (\(seqst,(s,t,phis)) -> (seqst,) <$> equivalentUnderConstraint s t phis (acceptable phis) (getValueCheck lc)) poss
  let equivs = do
        let poss = do
              s1@(CRSeq ((s, g1), _)) <- seqsS
              s2@(CRSeq ((t, g2), _)) <- seqsT
              let phis = conjGuards g1 g2
              return ((s1, s2), (s, t, phis))
        map
          ( \(seqst, (s, t, phis)) -> (seqst,) <$> equivalentUnderConstraint s t phis (acceptable phis)
          )
          poss
  witness <- findMs snd equivs
  case witness of
    Nothing -> return Nothing
    Just (seqs, _) -> return $ Just seqs
 where
  acceptable phis x = x `elem` varsGuard phis

findTrivial
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v, Sorted v)
  => [CRSeq (FId f) v]
  -> StateM (Maybe (CRSeq (FId f) v))
findTrivial = go
 where
  go [] = return Nothing
  -- go ((CRSeq []):rest) = go rest
  go (s@(CRSeq (cterm, _)) : rest) = do
    b <- trivialConstrainedEq cterm
    if b
      then return $ Just s
      else go rest

furtherStepsCRSeq
  :: (Monad m)
  => StepAnn
  -> (T.Term f v -> G.Guard f v -> m [CRSeq f v])
  -> CRSeq f v
  -> m [CRSeq f v]
furtherStepsCRSeq _ f (CRSeq ((t, c), [])) = f t c
furtherStepsCRSeq sAnn f (CRSeq ((t, c), (_, nextTerm) : seq)) = do
  followUpSeqs <- f t c
  return $ map (\newSeq -> combineCRSeqs newSeq sAnn $ CRSeq (nextTerm, seq)) followUpSeqs
