{- |
Module      : DevelopmentClosedness
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to analyze a given LCTRS for confluence
via (almost) development closedness and left-linearity.
-}
module Analysis.Confluence.DevelopmentClosedness where

import Analysis.Confluence.Confluence
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.LCTRS
import qualified Data.LCTRS.LCTRS as L
import Data.LCTRS.Position (position)
import Data.LCTRS.Term
import Data.LCTRS.VIdentifier (VId, freshV)
import Data.Maybe (
  catMaybes,
  isJust,
 )
import Data.Monad
import Data.SExpr
import Prettyprinter
import Rewriting.ConstrainedRewriting.MultiStepRewriting (multistepNSeq)
import Rewriting.CriticalPair
import Rewriting.CriticalPairSplitting (tryToClose)
import Rewriting.ParallelCriticalPair (renameFreshWithRenaming)
import Rewriting.Renaming (
  Renaming,
  innerRenaming,
 )

isDevelopmentClosed
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isDevelopmentClosed lc []
  | L.isLeftLinear lc = return $ Confluent (DevelopmentClosedness, "no critical pairs") -- cps <- computeCPs lc
isDevelopmentClosed lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isDevelopmentClosedCP lc) cps
      if all isJust closedOnes
        then return $ toResult DevelopmentClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isDevelopmentClosedCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isDevelopmentClosedCP lc CriticalPair{..} = do
  reductSeqs <- multistepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
  findTrivial reductSeqs
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

isAlmostDevelopmentClosed
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isAlmostDevelopmentClosed lc []
  | L.isLeftLinear lc = return $ Confluent (AlmostDevelopmentClosedness, "no critical pairs") -- cps <- computeCPs lc
isAlmostDevelopmentClosed lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isAlmostDevelopmentClosedCP lc) cps
      if all isJust closedOnes
        then return $ toResult AlmostDevelopmentClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isAlmostDevelopmentClosedCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isAlmostDevelopmentClosedCP lc cp@CriticalPair{..}
  | not (isOverlay cp) = isDevelopmentClosedCP lc cp
  | otherwise = do
      redsS <- multistepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
      redsT <-
        concat
          <$> mapM (furtherStepsCRSeq MulStep (curry $ multistepNSeq 1 (Just $ position [1]) ren fV lcR)) redsS
      findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

isAlmostDevelopmentClosedH
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isAlmostDevelopmentClosedH lc []
  | L.isLeftLinear lc = return $ Confluent (AlmostDevelopmentClosedness, "no critical pairs") -- cps <- computeCPs lc
isAlmostDevelopmentClosedH lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isAlmostDevelopmentClosedHCP lc) cps
      if all isJust closedOnes
        then return $ toResult AlmostDevelopmentClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isAlmostDevelopmentClosedHCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isAlmostDevelopmentClosedHCP lc cp@CriticalPair{..}
  | not (isOverlay cp) = isDevelopmentClosedCP lc cp
  | otherwise = do
      redsS <- multistepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
      redsT <-
        concat
          <$> mapM
            (furtherStepsCRSeq RefTraStep (curry $ multistepNSeq heuristic (Just $ position [1]) ren fV lcR))
            redsS
      findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

  heuristic = 5 -- min 10 $ length (rules lc)

toResult
  :: (Pretty f, Pretty v)
  => CRMethod
  -> [ ( CriticalPair (FId f) (VId v) (VId v)
       , [ ( CriticalPair (FId f) (VId v) (VId v)
           , CRSeq (FId f) (Renaming (VId v) (VId v))
           )
         ]
       )
     ]
  -> CRResult
toResult res closedOnes =
  Confluent
    ( res
    , show $
        vsep
          [ indent 2 $ printProofs cp proofs
          | (cp, proofs) <- closedOnes
          ]
    )
 where
  printProofs _ [] = error "DevelopmentClosedness.hs: CCP was not closed?"
  printProofs _ [(cp, proof)] = printCPClosing "*" cp proof
  printProofs origCCP proofs =
    vsep
      [ "*" <+> prettyCriticalPairFId origCCP
      , "is split into the following CCPs and closed as follows:"
      , indent 2 $ vsep [printCPClosing "**" cp proof | (cp, proof) <- proofs]
      ]

  printCPClosing label cp s1 =
    label
      <+> prettyCriticalPairFId cp
      <> line
      <> "which reaches the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s1)
