{- |
Module      : NewmansLemma
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the main functionality to check a given LCTRS if it satisfies the properties
to have confluence via Newmans Lemma. Currently it does not check for termination and for the result
to be true, the LCTRS must be termination.
-}
module Analysis.Confluence.NewmansLemma where

import Analysis.Confluence.Confluence (
  CRMethod (NewmansLemma),
  CRResult (Confluent, MaybeConfluent),
  findTrivial,
  furtherStepsCRSeq,
 )
import Analysis.Termination.CheckingTermination (
  defaultStrategy,
 )
import Analysis.Termination.Termination (SNInfoList, SNResult (Terminating))
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.LCTRS (
  CRSeq,
  LCTRS,
  StepAnn (RefTraStep),
  prettyCRSeqFId,
 )
import qualified Data.LCTRS.LCTRS as L
import Data.LCTRS.Position (position)
import Data.LCTRS.Term (cEq)
import Data.LCTRS.VIdentifier (VId, freshV)
import Data.Maybe (catMaybes, fromJust, isJust)
import Data.Monad (StateM)
import Data.SExpr (ToSExpr)
import Prettyprinter (Doc, Pretty (..), indent, line, vsep, (<+>))
import Rewriting.ConstrainedRewriting.SingleStepRewriting (rewriteNSeq)
import Rewriting.CriticalPair (
  CriticalPair (..),
  prettyCriticalPairFId,
 )
import Rewriting.CriticalPairSplitting (tryToClose)
import Rewriting.ParallelCriticalPair (renameFreshWithRenaming)
import Rewriting.Renaming (
  Renaming,
  innerRenaming,
 )

isCRbyNewmansLemma
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isCRbyNewmansLemma lc cps = do
  case cps of
    [] -> do
      sn <- isTerminating lc
      if isJust sn
        then
          return $
            Confluent
              ( NewmansLemma
              , show $
                  vsep
                    [ "Termination:"
                    , indent 2 $ pretty $ fromJust sn
                    , "Local Confluence:"
                    , indent 2 "no critical pairs"
                    ]
              )
        else return MaybeConfluent
    _ -> do
      sn <- isTerminating lc
      -- wcr <- foldrM (\cp isWCR -> (&& isWCR) <$> isLocallyConfluent lc cp) True cps
      -- wcr <- mapM (\cp -> fmap (cp,) <$> isLocallyConfluent lc cp) cps
      wcr <- mapM (\cp -> fmap (cp,) <$> tryToClose isLocallyConfluent lc cp) cps
      if isJust sn && all isJust wcr
        then
          return $
            Confluent
              ( NewmansLemma
              , show $
                  vsep
                    [ "Termination:"
                    , indent 2 $ pretty $ fromJust sn
                    , "Local Confluence:"
                    , indent 2 $ toResult $ catMaybes wcr
                    ]
              )
        else return MaybeConfluent

isLocallyConfluent
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isLocallyConfluent lc CriticalPair{..} = do
  redsSN <- rewriteNSeq heuristicS (Just $ position [0]) ren fV lcR constrainedEquation c
  redsTN <-
    concat
      <$> mapM
        (furtherStepsCRSeq RefTraStep (rewriteNSeq heuristicT (Just $ position [1]) ren fV lcR))
        redsSN
  findTrivial redsTN
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  constrainedEquation = cEq s t

  heuristicS = 50 -- min 10 $ length (rules lc)
  heuristicT = 50 -- min 10 $ length (rules lc)
  lcR = L.rename innerRenaming lc

-- furtherSteps _ (CRSeq []) = return []
-- furtherSteps f (CRSeq ((t,c):seq)) = do
--   followUpSeqs <- f t c
--   return $ map (`combineSeqs` CRSeq seq) followUpSeqs

-- combineSeqs (CRSeq seq1) (CRSeq seq2) = CRSeq $ seq1 ++ seq2

isTerminating
  :: (Ord v, Ord f, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (Maybe (SNInfoList (FId f) (VId v)))
isTerminating lc =
  go $ defaultStrategy lc
 where
  -- [ terminatingByValueCriterion
  -- , terminatingByRPOwithDP
  -- , terminatingByMatrixIntersDP 2 Nothing
  -- , terminatingByMatrixIntersDP 3 Nothing
  -- , terminatingByReductionPair
  --     [ rpoRP
  --     , valueCriterionRP
  --     , matrixInterpretationsRP 1 Nothing
  --     , matrixInterpretationsRP 2 Nothing
  --     , matrixInterpretationsRP 3 Nothing
  --     ]
  -- ]

  -- go [] = return Nothing
  -- go (m : ms) = do
  --   (res, proof) <- m lc
  --   if res == Terminating
  --     then return $ Just proof
  --     else go ms

  go [] = return Nothing
  go (m : ms) = do
    (res, proof) <- m
    if res == Terminating
      then return $ Just proof
      else go ms

toResult
  :: (Pretty f, Pretty v)
  => [ ( CriticalPair (FId f) (VId v) (VId v)
       , [ ( CriticalPair (FId f) (VId v) (VId v)
           , CRSeq (FId f) (Renaming (VId v) (VId v))
           )
         ]
       )
     ]
  -> Doc ann
toResult closedOnes =
  vsep
    [ indent 2 $ printProofs cp proofs
    | (cp, proofs) <- closedOnes
    ]
 where
  printProofs _ [] = error "NewmansLemma.hs: CCP was not joined?"
  printProofs _ [(cp, proof)] = printCPClosing "*" cp proof
  printProofs origCCP proofs =
    vsep
      [ "*" <+> prettyCriticalPairFId origCCP
      , "is split into the following CCPs and joined as follows:"
      , indent 2 $ vsep [printCPClosing "**" cp proof | (cp, proof) <- proofs]
      ]

  printCPClosing label cp s1 =
    label
      <+> prettyCriticalPairFId cp
      <> line
      <> "which reaches the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s1)
