{- |
Module      : OneParallelClosedness
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to check a given LCTRS for
partal parallel closedness.
-}
module Analysis.Confluence.OneParallelClosedness (isOneParallelClosed, isOneParallelClosedCP, isOneParallelClosedHCP)
where

import Analysis.Confluence.Confluence
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.LCTRS
import qualified Data.LCTRS.LCTRS as L
import Data.LCTRS.Position (position)
import Data.LCTRS.Term
import Data.LCTRS.VIdentifier (VId, freshV)
import Data.Maybe (
  catMaybes,
  isJust,
 )
import Data.Monad
import Data.SExpr
import Prettyprinter
import Rewriting.ConstrainedRewriting.ParallelStepRewriting (parallelRewriteStepNSeq)
import Rewriting.CriticalPair
import Rewriting.CriticalPairSplitting (tryToClose)
import Rewriting.ParallelCriticalPair (renameFreshWithRenaming)
import Rewriting.Renaming (
  Renaming,
  innerRenaming,
 )

-- isOneParallelClosed
--   :: (Ord v, Ord f, Pretty v, ToSExpr f, ToSExpr v, Pretty f)
--   => LCTRS (FId f) (VId v)
--   -> [CriticalPair (FId f) (VId v) (VId v)]
--   -> StateM
--       (Maybe [(CriticalPair (FId f) (VId v) (VId v), CRSeq (FId f) (Renaming (VId v) (VId v)))])
isOneParallelClosed
  :: (Pretty v, Pretty f, Ord v, Ord f, ToSExpr v, ToSExpr f)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM
      ( Maybe
          [ ( CriticalPair (FId f) (VId v) (VId v)
            , [(CriticalPair (FId f) (VId v) (VId v), CRSeq (FId f) (Renaming (VId v) (VId v)))]
            )
          ]
      )
isOneParallelClosed _ [] = return Nothing
isOneParallelClosed lc cps = do
  closedness <-
    mapM (tryToClose (\lc cp -> isOneParallelClosedCP lc cp `alt` isOneParallelClosedHCP lc cp) lc) cps
  if all isJust closedness
    then return . Just . zip cps $ catMaybes closedness
    else return Nothing
 where
  alt c1 c2 = do
    v <- c1
    case v of
      Nothing -> c2
      jseq -> return jseq

isOneParallelClosedCP
  :: (Ord v, Ord f, Pretty v, ToSExpr f, ToSExpr v, Pretty f)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isOneParallelClosedCP lc CriticalPair{..} = do
  redsS <- parallelRewriteStepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
  redsT <-
    concat
      <$> mapM
        ( furtherStepsCRSeq
            (ParStep Nothing)
            (curry $ parallelRewriteStepNSeq 1 (Just $ position [1]) ren fV lcR)
        )
        redsS
  findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

isOneParallelClosedHCP
  :: (Ord v, Ord f, Pretty v, ToSExpr f, ToSExpr v, Pretty f)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isOneParallelClosedHCP lc CriticalPair{..} = do
  redsS <- parallelRewriteStepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
  redsT <-
    concat
      <$> mapM
        ( furtherStepsCRSeq
            RefTraStep
            (curry $ parallelRewriteStepNSeq heuristic (Just $ position [1]) ren fV lcR)
        )
        redsS
  findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

  heuristic = 5 -- min 10 $ length (rules lc)
