{- |
Module      : Orthogonality
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to analyze a given LCTRS for confluence
via (weak) orthogonality and left-linearity.
-}
module Analysis.Confluence.Orthogonality where

import Analysis.Confluence.Confluence
import Data.LCTRS.FIdentifier (
  FId,
 )
import Data.LCTRS.LCTRS
import Data.LCTRS.Rule (prettyCTermFId)
import Data.LCTRS.Term (cEq)
import Data.LCTRS.VIdentifier (
  VId,
 )
import Data.Monad
import Data.SExpr
import Prettyprinter
import Rewriting.CriticalPair

isOrthogonal
  :: (Ord v, Ord f)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isOrthogonal lc [] | isLeftLinear lc = return $ Confluent (Orthogonality, "no critical pairs")
isOrthogonal _ _ =
  return MaybeConfluent

isWeaklyOrthogonal
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isWeaklyOrthogonal stdLC cps | isLeftLinear stdLC = do
  -- cps <- computeCPs stdLC
  case cps of
    [] -> return $ Confluent (WeakOrthogonality, "no critical pairs")
    _ -> do
      isCRbyWO <- and <$> mapM isTrivialCP cps
      if not isCRbyWO
        then return MaybeConfluent
        else
          return $
            Confluent
              ( WeakOrthogonality
              , show $
                  vsep
                    [ "*"
                      <+> prettyCriticalPairFId cp
                      <> line
                      <> "but it is a trivial constrained equation"
                      <> line
                      <> indent 2 (prettyCTermFId (cEq (inner cp) (outer cp), constraint cp))
                    | -- <+> "~"
                    -- <+> prettyCTermFId (outer cp, constraint cp)
                    -- <> "but is trivial as"
                    -- <> line
                    -- <> indent 2 (prettyCTermFId (inner cp, constraint cp))
                    --   <+> "~"
                    --   <+> prettyCTermFId (outer cp, constraint cp)
                    -- ++ " "
                    -- ++ toSExpr (cpGuard cp)
                    cp <- cps
                    ]
              )
isWeaklyOrthogonal _ _ =
  return MaybeConfluent
