{- |
Module      : ParallelClosedness
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to analyze a given LCTRS for confluence
via (almost) parallel closedness and left-linearity.
-}
module Analysis.Confluence.ParallelClosedness where

import Analysis.Confluence.Confluence
import Data.LCTRS.LCTRS
import qualified Data.LCTRS.LCTRS as L
import Data.LCTRS.Term
import Data.Maybe (
  catMaybes,
  isJust,
 )
import Data.Monad

import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.Position (position)
import Data.LCTRS.VIdentifier (VId, freshV)
import Data.SExpr
import Prettyprinter
import Rewriting.ConstrainedRewriting.ParallelStepRewriting (parallelRewriteStepNSeq)
import Rewriting.CriticalPair
import Rewriting.CriticalPairSplitting (tryToClose)
import Rewriting.ParallelCriticalPair (renameFreshWithRenaming)
import Rewriting.Renaming (
  Renaming,
  innerRenaming,
 )

isParallelClosed
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isParallelClosed lc []
  | L.isLeftLinear lc = return $ Confluent (ParallelClosedness, "no critical pairs") -- cps <- computeCPs lc
isParallelClosed lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isParallelClosedCP lc) cps
      if all isJust closedOnes
        then return $ toResult ParallelClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isParallelClosedCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isParallelClosedCP lc CriticalPair{..} = do
  reductSeqs <- parallelRewriteStepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
  findTrivial reductSeqs
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

isAlmostParallelClosed
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isAlmostParallelClosed lc []
  | L.isLeftLinear lc = return $ Confluent (AlmostParallelClosedness, "no critical pairs") -- cps <- computeCPs lc
isAlmostParallelClosed lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isAlmostParallelClosedCP lc) cps
      if all isJust closedOnes
        then return $ toResult AlmostParallelClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isAlmostParallelClosedCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isAlmostParallelClosedCP lc cp@CriticalPair{..}
  | not (isOverlay cp) = isParallelClosedCP lc cp
  | otherwise = do
      redsS <- parallelRewriteStepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
      redsT <-
        concat
          <$> mapM
            ( furtherStepsCRSeq
                (ParStep Nothing)
                (curry $ parallelRewriteStepNSeq 1 (Just $ position [1]) ren fV lcR)
            )
            redsS
      findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

isAlmostParallelClosedH
  :: (Ord v, Ord f, Pretty v, Pretty f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isAlmostParallelClosedH lc []
  | L.isLeftLinear lc = return $ Confluent (AlmostParallelClosedness, "no critical pairs") -- cps <- computeCPs lc
isAlmostParallelClosedH lc cps
  | L.isLeftLinear lc = do
      closedOnes <- mapM (tryToClose isAlmostParallelClosedHCP lc) cps
      if all isJust closedOnes
        then return $ toResult AlmostParallelClosedness $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isAlmostParallelClosedHCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM (Maybe (CRSeq (FId f) (Renaming (VId v) (VId v))))
isAlmostParallelClosedHCP lc cp@CriticalPair{..}
  | not (isOverlay cp) = isParallelClosedCP lc cp
  | otherwise = do
      redsS <- parallelRewriteStepNSeq 1 (Just $ position [0]) ren fV lcR (cEq s t, c)
      redsT <-
        concat
          <$> mapM
            ( furtherStepsCRSeq
                RefTraStep
                (curry $ parallelRewriteStepNSeq heuristic (Just $ position [1]) ren fV lcR)
            )
            redsS
      findTrivial redsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  lcR = L.rename innerRenaming lc

  heuristic = 5 -- min 10 $ length (rules lc)

toResult
  :: (Pretty f, Pretty v)
  => CRMethod
  -> [ ( CriticalPair (FId f) (VId v) (VId v)
       , [ ( CriticalPair (FId f) (VId v) (VId v)
           , CRSeq (FId f) (Renaming (VId v) (VId v))
           )
         ]
       )
     ]
  -> CRResult
toResult res closedOnes =
  Confluent
    ( res
    , show $
        vsep
          [ indent 2 $ printProofs cp proofs
          | (cp, proofs) <- closedOnes
          ]
    )
 where
  printProofs _ [] = error "ParallelClosedness.hs: CCP was not closed?"
  printProofs _ [(cp, proof)] = printCPClosing "*" cp proof
  printProofs origCCP proofs =
    vsep
      [ "*" <+> prettyCriticalPairFId origCCP
      , "is split into the following CCPs and closed as follows:"
      , indent 2 $ vsep [printCPClosing "**" cp proof | (cp, proof) <- proofs]
      ]

  printCPClosing label cp s1 =
    label
      <+> prettyCriticalPairFId cp
      <> line
      <> "which reaches the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s1)
