{- |
Module      : StrongClosedness
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to analyze a given LCTRS for confluence
via strongly closedness and left-linearity.
-}
module Analysis.Confluence.StrongClosedness where

import Analysis.Confluence.Confluence
import Data.LCTRS.FIdentifier
import Data.LCTRS.LCTRS
import qualified Data.LCTRS.LCTRS as L
import Data.LCTRS.Position (position)
import Data.LCTRS.Term
import Data.LCTRS.VIdentifier
import Data.Maybe (catMaybes, isJust)
import Data.Monad
import Data.SExpr
import Prettyprinter
import Rewriting.ConstrainedRewriting.SingleStepRewriting (rewriteNSeq)
import Rewriting.CriticalPair
import Rewriting.CriticalPairSplitting (tryToClose)
import Rewriting.ParallelCriticalPair (renameFreshWithRenaming)
import Rewriting.Renaming (
  Renaming,
  innerRenaming,
 )

isStrongClosed
  :: (Pretty f, Pretty v, Ord v, Ord f, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
isStrongClosed lc []
  | L.isLinear lc = return $ Confluent (StrongClosedness, "no critical pairs") -- cps <- computeCPs lc
isStrongClosed lc cps
  | L.isLinear lc = do
      closedOnes <- mapM (tryToClose isStrongClosedCP lc) cps
      if all isJust closedOnes
        then return $ toResult $ zip cps $ catMaybes closedOnes
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

isStrongClosedCP
  :: (Ord v, Ord f, Pretty f, Pretty v, ToSExpr f, ToSExpr v)
  => LCTRS (FId f) (VId v)
  -> CriticalPair (FId f) (VId v) (VId v)
  -> StateM
      ( Maybe
          ( CRSeq (FId f) (Renaming (VId v) (VId v))
          , CRSeq (FId f) (Renaming (VId v) (VId v))
          )
      )
isStrongClosedCP lc CriticalPair{..} = do
  redsSN <-
    rewriteNSeq
      heuristicS
      (Just $ position [0])
      ren
      fV
      lcR
      constrainedEquation
      c
  redsTN <-
    rewriteNSeq
      heuristicT
      (Just $ position [1])
      ren
      fV
      lcR
      constrainedEquation
      c
  redsTR <-
    concat
      <$> mapM (furtherStepsCRSeq SinStep (rewriteNSeq 1 (Just $ position [1]) ren fV lcR)) redsSN
  redsSR <-
    concat
      <$> mapM (furtherStepsCRSeq SinStep (rewriteNSeq 1 (Just $ position [0]) ren fV lcR)) redsTN
  equivsS <- findTrivial redsTR
  equivsT <- findTrivial redsSR
  return $ (,) <$> equivsS <*> equivsT
 where
  ren = renameFreshWithRenaming

  fV s i = innerRenaming $ freshV s i

  s = inner
  t = outer
  c = constraint

  constrainedEquation = cEq s t

  heuristicS = 5 -- min 10 $ length (rules lc)
  heuristicT = 5 -- min 10 $ length (rules lc)
  lcR = rename innerRenaming lc

toResult
  :: (Pretty f, Pretty v)
  => [ ( CriticalPair (FId f) (VId v) (VId v)
       , [ ( CriticalPair (FId f) (VId v) (VId v)
           , ( CRSeq (FId f) (Renaming (VId v) (VId v))
             , CRSeq (FId f) (Renaming (VId v) (VId v))
             )
           )
         ]
       )
     ]
  -> CRResult
toResult closedOnes =
  Confluent
    ( StrongClosedness
    , show $
        vsep
          [ indent 2 $ printProofs cp proofs
          | (cp, proofs) <- closedOnes
          ]
    )
 where
  printProofs _ [] = error "StrongClosedness.hs: CCP was not closed?"
  printProofs _ [(cp, proof)] = printCPClosing "*" cp proof
  printProofs origCCP proofs =
    vsep
      [ "*" <+> prettyCriticalPairFId origCCP
      , "is split into the following CCPs and closed as follows:"
      , indent 2 $ vsep [printCPClosing "**" cp proof | (cp, proof) <- proofs]
      ]

  printCPClosing label cp (s1, s2) =
    label
      <+> prettyCriticalPairFId cp
      <> line
      <> "which reaches the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s1)
      <> line
      <> "and the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s2)
