{- |
Module      : Toyama81
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the functionality to check if a given LCTRS
satisfies the result of Toyama 81.
-}
module Analysis.Confluence.Toyama81 (toyama81Applies)
where

import Analysis.Confluence.Confluence
import Analysis.Confluence.OneParallelClosedness (isOneParallelClosed)
import Analysis.Confluence.TwoParallelClosedness (isTwoParallelClosed, varsUnderPoss)
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.Guard (varsGuard)
import Data.LCTRS.LCTRS
import Data.LCTRS.VIdentifier (VId)
import Data.Maybe (fromJust, isJust)
import Data.Monad (StateM)
import Data.SExpr (ToSExpr)
import qualified Data.Set as S
import Prettyprinter (Doc, Pretty, encloseSep, indent, line, pretty, vsep, (<+>))
import Rewriting.CriticalPair (CriticalPair, prettyCriticalPairFId)
import Rewriting.ParallelCriticalPair (
  ParallelCriticalPair (..),
  prettyParallelCriticalPair,
 )
import Rewriting.Renaming (
  Renaming,
 )

toyama81Applies
  :: (Ord v, Ord f, Pretty v, ToSExpr f, ToSExpr v, Pretty f)
  => LCTRS (FId f) (VId v)
  -> [CriticalPair (FId f) (VId v) (VId v)]
  -> [ParallelCriticalPair (FId f) (VId v) (VId v)]
  -> StateM CRResult
toyama81Applies lc [] []
  | isLeftLinear lc = return $ Confluent (Toyama81, "no critical pairs and no parallel critcal pairs")
toyama81Applies lc cps pcps
  | isLeftLinear lc = do
      oneClosedness <- isOneParallelClosed lc cps
      twoClosedness <- isTwoParallelClosed lc pcps
      if isJust oneClosedness && isJust twoClosedness
        then
          return $
            Confluent
              ( Toyama81
              , show $
                  prettyOneClosedness (fromJust oneClosedness)
                    <> line
                    <> vsep
                      ( [ "*"
                          <+> prettyParallelCriticalPair pcp
                          <> line
                          <> "satisfies the variable condition"
                          <> line
                          <> indent 2 (prettyVarCondition pcp s)
                          <> line
                          <> "and reaches the trivial constrained equation"
                          <> line
                          <> indent 2 (prettyCRSeqFId s)
                        | let proofs = fromJust twoClosedness
                        , (pcp, s) <- proofs
                        ]
                      )
              )
        else return MaybeConfluent
  | otherwise = return MaybeConfluent

prettyOneClosedness
  :: (Pretty f, Pretty v)
  => [ ( CriticalPair (FId f) (VId v) (VId v)
       , [ ( CriticalPair (FId f) (VId v) (VId v)
           , CRSeq (FId f) (Renaming (VId v) (VId v))
           )
         ]
       )
     ]
  -> Doc ann
prettyOneClosedness closedOnes =
  vsep
    [ printCPProofs cp proofs
    | (cp, proofs) <- closedOnes
    ]
 where
  printCPProofs _ [] = error "Toyama81.hs: CCP was not closed?"
  printCPProofs _ [(cp, proof)] = printCPClosing "*" cp proof
  printCPProofs origCCP proofs =
    vsep
      [ "*" <+> prettyCriticalPairFId origCCP
      , "is split into the following CCPs and closed as follows:"
      , indent 2 $ vsep [printCPClosing "**" cp proof | (cp, proof) <- proofs]
      ]

  printCPClosing label cp s =
    label
      <+> prettyCriticalPairFId cp
      <> line
      <> "which reaches the trivial constrained equation"
      <> line
      <> indent 2 (prettyCRSeqFId s)

prettySet :: (Pretty v) => S.Set (Renaming v v) -> Doc ann
prettySet = encloseSep "{" "}" ", " . map pretty . S.toList

prettyVarCondition
  :: (Ord v, Ord f, Pretty v) => ParallelCriticalPair f v v -> CRSeq f (Renaming v v) -> Doc ann
prettyVarCondition ParallelCriticalPair{..} (CRSeq (_, [])) =
  let varsC = S.fromList $ varsGuard constraint
  in  "{}"
        <+> "⊆"
        <+> prettySet (varsUnderPoss (`S.notMember` varsC) top (S.fromList $ map fst innerRules))
prettyVarCondition ParallelCriticalPair{..} (CRSeq ((b, bC), ((p, _) : _))) =
  let
    varsC = S.fromList $ varsGuard constraint
    varsBC = S.fromList $ varsGuard bC
  in
    prettySet (varsUnderPoss (`S.notMember` varsBC) b (getPoss p))
      <+> "⊆"
      <+> prettySet (varsUnderPoss (`S.notMember` varsC) top (S.fromList $ map fst innerRules))
 where
  getPoss (ParStep (Just poss)) = poss
  getPoss _ = error "Toyama81.hs: Cannot extract information about variable condition."
