{-# LANGUAGE ScopedTypeVariables #-}

{- |
Module      : CheckingTermination
Description :
Copyright   : (c) Jonas Schöpf, 2024
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable

This module provides functions for checking termination.
-}
module Analysis.Termination.CheckingTermination where

import Analysis.Termination.ConstrainedReductionOrder (rpoRP, rpoStrict)
import Analysis.Termination.DependencyGraph (
  approximateDPGraph,
  stronglyConnComps,
  transformDPProblemWithInfo,
 )
import Analysis.Termination.DependencyPairs (
  DPProblem (DPProblem, strictrules, weakrules),
  dpproblem,
 )
import Analysis.Termination.MatrixInterpretations (
  Dimension,
  matrixInterpretations,
  matrixInterpretationsRP,
 )
import Analysis.Termination.MatrixInterpretationsBV (
  Bits,
  matrixInterpretationsBV,
  matrixInterpretationsBVRP,
 )
import Analysis.Termination.MatrixInterpretationsConstraints (
  EntryUpperBound,
  matrixInterpretationsCons,
  matrixInterpretationsConsRP,
 )
import Analysis.Termination.ReductionPair (allRhsSubtermsGroundLogicalTerms, applyReductionPair)
import Analysis.Termination.SubtermCriterion (subtermCriterionRP)
import Analysis.Termination.Termination (
  SNInfo (..),
  SNInfoList (SNInfoList),
  SNResult (MaybeTerminating, NonTerminating, Terminating),
  emptyPrecedence,
 )
import Analysis.Termination.ValueCriterion (valueCriterionRP)
import Analysis.Termination.ValueCriterionPol (valueCriterionPolRP)
import Control.Monad.IO.Class (liftIO)
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.LCTRS (LCTRS, definesToSExpr, getRules)
import Data.LCTRS.VIdentifier (VId)
import Data.List (partition)
import Data.Monad (
  Error (MayError, SevError),
  FreshM,
  StateM,
  evalStateM,
  freshI,
 )
import Data.SExpr (ToSExpr)
import qualified Data.Set as S
import Prettyprinter (Pretty (..))
import SimpleSMT (Solver)
import Utils (asyncs, lenSensitiveZip, waitF)

----------------------------------------------------------------------------------------------------
-- main functionality
----------------------------------------------------------------------------------------------------

analyzeTermination
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => [Solver]
  -> Maybe [LCTRS (FId f) (VId v) -> StateM (SNResult, SNInfoList (FId f) (VId v))]
  -> Bool
  -> LCTRS (FId f) (VId v)
  -> FreshM (SNResult, SNInfoList (FId f) (VId v))
analyzeTermination solvers strategies debugFlag lctrs = do
  next <- freshI
  let collectedMethods = collectMethods lctrs strategies
  let matchSolverMethod = lenSensitiveZip collectedMethods solvers
  let computations =
        [ evalStateM solver (definesToSExpr lctrs) debugFlag next method
        | (method, solver) <- matchSolverMethod
        ]
  jobs <- liftIO $ asyncs computations
  result <- liftIO $ waitF analyzedSN (Right (MaybeTerminating, SNInfoList [None])) $ S.fromList jobs
  return $ handleError debugFlag result

collectMethods
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> Maybe [LCTRS (FId f) (VId v) -> StateM (SNResult, SNInfoList (FId f) (VId v))]
  -> [StateM (SNResult, SNInfoList (FId f) (VId v))]
collectMethods lctrs = maybe (defaultStrategy lctrs) applyThem
 where
  applyThem = map ($ lctrs)

defaultStrategy
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> [StateM (SNResult, SNInfoList (FId f) (VId v))]
defaultStrategy lctrs =
  [ terminatingByReductionPair
      [ valueCriterionRP
      , subtermCriterionRP
      , rpoRP
      , valueCriterionPolRP
      ]
      lctrs
  ]

----------------------------------------------------------------------------------------------------
-- termination methods
----------------------------------------------------------------------------------------------------

terminatingByDependencyGraph
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByDependencyGraph lctrs = do
  dpgraph <- approximateDPGraph lctrs
  let sccs = stronglyConnComps dpgraph
  if null sccs
    then return (Terminating, SNInfoList [DpGraph (dpproblem lctrs) (dpgraph, sccs)])
    else return (MaybeTerminating, SNInfoList [None])

solveWithDPGraph
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => (DPProblem (FId f) (VId v) -> StateM (Maybe (DPProblem (FId f) (VId v), SNInfo (FId f) (VId v))))
  -> DPProblem (FId f) (VId v)
  -> StateM (Maybe [SNInfo (FId f) (VId v)])
solveWithDPGraph _ dpp | null $ strictrules dpp = return $ Just [NoDependencyPairs]
solveWithDPGraph solve dpp = do
  (info, sccs) <- transformDPProblemWithInfo dpp
  maybeSomeProgress <- mapM solve sccs
  case sequence maybeSomeProgress of
    Nothing -> return Nothing
    Just everyWhereProgress -> do
      let (solved, unsolved) = partition (null . strictrules . fst) everyWhereProgress
      recs <-
        mapM
          ( \(undpp, sninfo) -> fmap (sninfo :) <$> solveWithDPGraph solve undpp
          )
          unsolved
      case sequence recs of
        Nothing -> return Nothing
        Just remainingSolutions -> return $ Just $ DpGraph dpp info : map snd solved <> concat remainingSolutions

terminatingByValueCriterion
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByValueCriterion lctrs =
  let dpp = dpproblem lctrs
  in  if null $ strictrules dpp
        then return (Terminating, SNInfoList [NoDependencyPairs])
        else do
          let solve givenDPP = applyReductionPair givenDPP valueCriterionRP
          result <- solveWithDPGraph solve dpp
          case result of
            Nothing -> return (MaybeTerminating, SNInfoList [None])
            Just sninfo -> return (Terminating, SNInfoList sninfo)

terminatingByValueCriterionPol
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByValueCriterionPol lctrs =
  let dpp = dpproblem lctrs
  in  if null $ strictrules dpp
        then return (Terminating, SNInfoList [NoDependencyPairs])
        else do
          let solve givenDPP = applyReductionPair givenDPP valueCriterionPolRP
          result <- solveWithDPGraph solve dpp
          case result of
            Nothing -> return (MaybeTerminating, SNInfoList [None])
            Just sninfo -> return (Terminating, SNInfoList sninfo)

terminatingBySubtermCriterion
  :: forall f v
   . (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingBySubtermCriterion lctrs =
  let dpp = dpproblem lctrs
  in  if null $ strictrules dpp
        then return (Terminating, SNInfoList [NoDependencyPairs])
        else do
          let solve givenDPP = applyReductionPair givenDPP subtermCriterionRP
          result <- solveWithDPGraph solve dpp
          case result of
            Nothing -> return (MaybeTerminating, SNInfoList [None])
            Just sninfo -> return (Terminating, SNInfoList sninfo)

terminatingByRPO
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByRPO lctrs = do
  (res, info) <- rpoStrict $ getRules lctrs
  case info of
    Just i -> return (res, SNInfoList [RpoPrecedence i])
    Nothing -> return (res, SNInfoList [RpoPrecedence emptyPrecedence])

terminatingByRPOwithDP
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByRPOwithDP lctrs =
  let dpp = dpproblem lctrs
  in  if null $ strictrules dpp
        then return (Terminating, SNInfoList [NoDependencyPairs])
        else do
          let solve givenDPP = applyReductionPair givenDPP rpoRP
          result <- solveWithDPGraph solve dpp
          case result of
            Nothing -> return (MaybeTerminating, SNInfoList [None])
            Just sninfo -> return (Terminating, SNInfoList sninfo)

----------------------------------------------------------------------------------------------------
-- termination methods based on interpretations (mostly experimental)
----------------------------------------------------------------------------------------------------

terminatingByMatrixInters
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Maybe EntryUpperBound
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByMatrixInters dim upperB lctrs = do
  result <- matrixInterpretations dim upperB (getRules lctrs)
  case result of
    (Terminating, Just inters) ->
      return
        (Terminating, SNInfoList [MatrixInterpretation (DPProblem (getRules lctrs) []) inters])
    _ -> return (MaybeTerminating, SNInfoList [None])

terminatingByMatrixIntersDP
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Maybe EntryUpperBound
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByMatrixIntersDP dim upperB lctrs = do
  let dpp = dpproblem lctrs
  if null $ strictrules dpp
    then return (Terminating, SNInfoList [NoDependencyPairs])
    else do
      let solve givenDPP = applyReductionPair givenDPP $ matrixInterpretationsRP dim upperB
      result <- solveWithDPGraph solve dpp
      case result of
        Nothing -> return (MaybeTerminating, SNInfoList [None])
        Just sninfo -> return (Terminating, SNInfoList sninfo)

terminatingByBVMatrixInters
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Bits
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByBVMatrixInters dim bits lctrs = do
  result <- matrixInterpretationsBV dim bits (getRules lctrs)
  case result of
    (Terminating, Just inters) ->
      return
        (Terminating, SNInfoList [MatrixInterpretation (DPProblem (getRules lctrs) []) inters])
    _ -> return (MaybeTerminating, SNInfoList [None])

terminatingByBVMatrixIntersDP
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Bits
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByBVMatrixIntersDP dim bits lctrs = do
  let dpp = dpproblem lctrs
  if null $ strictrules dpp
    then return (Terminating, SNInfoList [NoDependencyPairs])
    else do
      let solve givenDPP = applyReductionPair givenDPP $ matrixInterpretationsBVRP dim bits
      result <- solveWithDPGraph solve dpp
      case result of
        Nothing -> return (MaybeTerminating, SNInfoList [None])
        Just sninfo -> return (Terminating, SNInfoList sninfo)

terminatingByConsMatrixInters
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Maybe EntryUpperBound
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByConsMatrixInters dim uBound lctrs = do
  result <- matrixInterpretationsCons dim uBound (getRules lctrs)
  case result of
    (Terminating, Just inters) ->
      return
        (Terminating, SNInfoList [MatrixInterpretation (DPProblem (getRules lctrs) []) inters])
    _ -> return (MaybeTerminating, SNInfoList [None])

terminatingByConsMatrixIntersDP
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => Dimension
  -> Maybe EntryUpperBound
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByConsMatrixIntersDP dim uBound lctrs = do
  let dpp = dpproblem lctrs
  if null $ strictrules dpp
    then return (Terminating, SNInfoList [NoDependencyPairs])
    else do
      let solve givenDPP = applyReductionPair givenDPP $ matrixInterpretationsConsRP dim uBound
      result <- solveWithDPGraph solve dpp
      case result of
        Nothing -> return (MaybeTerminating, SNInfoList [None])
        Just sninfo -> return (Terminating, SNInfoList sninfo)

----------------------------------------------------------------------------------------------------
-- termination methods on reduction pairs
----------------------------------------------------------------------------------------------------

terminatingByReductionPair
  :: (Ord f, Ord v, ToSExpr f, ToSExpr v, Pretty f, Pretty v)
  => [ DPProblem (FId f) (VId v)
     -> StateM (DPProblem (FId f) (VId v), Maybe (SNInfo (FId f) (VId v)))
     ]
  -> LCTRS (FId f) (VId v)
  -> StateM (SNResult, SNInfoList (FId f) (VId v))
terminatingByReductionPair rps lctrs = do
  let dpp = dpproblem lctrs
  if null $ strictrules dpp
    then return (Terminating, SNInfoList [NoDependencyPairs])
    else do
      let solve givenDPP =
            let givenDPP' = needWeakRules givenDPP
            in  findApplied [applyReductionPair givenDPP' rp | rp <- rps]
      result <- solveWithDPGraph solve dpp
      case result of
        Nothing -> return (MaybeTerminating, SNInfoList [None])
        Just sninfo -> return (Terminating, SNInfoList sninfo)
 where
  -- NOTE: IMPORTANT
  -- be aware of the third condition which demands that the weak
  -- part of the reduction pair orients all calculation rules weakly
  -- rps = [rpoRP, valueCriterionRP] ++ map matrixInterpretationsRP [1 .. 3]
  needWeakRules dpp@DPProblem{..}
    | all allRhsSubtermsGroundLogicalTerms strictrules = DPProblem strictrules []
    | otherwise = dpp

  findApplied [] = return Nothing
  findApplied (computation : remaining) = do
    value <- computation
    case value of
      Nothing -> findApplied remaining
      Just result -> return $ Just result

----------------------------------------------------------------------------------------------------
-- auxiliary methods
----------------------------------------------------------------------------------------------------

handleError :: Bool -> Either Error (SNResult, SNInfoList f v) -> (SNResult, SNInfoList f v)
handleError False (Left (MayError _)) = (MaybeTerminating, SNInfoList [None])
-- for debugging if the result is MAYBE because of an non severe error
handleError True (Left (MayError s)) = error $ "A non-severe error occured in the termination analysis " ++ s
handleError False (Left (SevError _)) = (MaybeTerminating, SNInfoList [None])
-- for debugging if the result is MAYBE because of an error
handleError True (Left (SevError s)) = error $ "A severe error occured in the termination analysis " ++ s
handleError _ (Right r) = r

analyzedSN :: Either Error (SNResult, SNInfoList f v) -> Bool
analyzedSN (Right (Terminating, _)) = True
analyzedSN (Right (NonTerminating, _)) = True
-- for debugging if the result is MAYBE because of an non severe error
-- analyzedSN (Left (MayError e)) = error $ "A non-severe error occured in the termination analysis " ++ e -- Debugging
analyzedSN (Left (SevError e)) = error $ "A severe error occured in the termination analysis " ++ e -- Debugging
analyzedSN _ = False
