module Analysis.Termination.DependencyPairs where

import Data.LCTRS.FIdentifier (FId (FId, FIdDp))
import Data.LCTRS.LCTRS (LCTRS, definedSyms, getRules)
import Data.LCTRS.Rule (Rule, createRule, funs, guard, lhs, prettyRule, rhs)
import Data.LCTRS.Sort (dpSort, inSort, sortAnnotation)
import Data.LCTRS.Term (Term (..), fun, getRoot, subterms)
import Data.LCTRS.VIdentifier (VId)
import qualified Data.Set as S
import Prettyprinter (Pretty (..), encloseSep, indent, vsep)

data DPProblem f v = DPProblem
  { strictrules :: [Rule f v]
  , weakrules :: [Rule f v]
  }
  deriving (Eq)

instance (Pretty f, Pretty v) => Pretty (DPProblem f v) where
  pretty DPProblem{..} =
    vsep
      [ "dependency pairs:"
      , indent 2 $ encloseSep "{" "}" ", " $ map prettyRule strictrules
      , "rules:"
      , indent 2 $ encloseSep "{" "}" ", " $ map prettyRule weakrules
      ]

dpproblem :: (Ord f) => LCTRS (FId f) (VId v) -> DPProblem (FId f) (VId v)
dpproblem lctrs = DPProblem (computeDPs lctrs) (getRules lctrs)

computeDPs :: (Ord f) => LCTRS (FId f) (VId v) -> [Rule (FId f) (VId v)]
computeDPs lctrs = do
  r <- rules
  st <- subterms $ rhs r
  case getRoot st of
    Left stf
      | stf `S.member` defSyms ->
          return $ createRule (setRootDpSym . lhs $ r) (setRootDpSym st) (guard r)
    _ -> []
 where
  rules = getRules lctrs
  defSyms = definedSyms lctrs

dpSymbols :: Rule (FId f) (VId v) -> [FId f]
dpSymbols rule = filter isDPSym $ funs rule

isDPSym :: FId f -> Bool
isDPSym (FIdDp _ _) = True
isDPSym _ = False

setRootDpSym :: Term (FId f) (VId v) -> Term (FId f) (VId v)
setRootDpSym (Var _) = error "DependencyPairs.hs: DP symbols for variables are not supported."
setRootDpSym (Fun typ f args) = fun typ (newf f) args
 where
  newf (FId f sa) = FIdDp f $ sortAnnotation (inSort sa) dpSort
  newf fid@(FIdDp _ _) = fid
  newf _ = error "DependencyPairs.hs: Cannot transform unknown symbol into DP symbol."
