{- |
Module      : FIdentifier
Description :
Copyright   : (c) Jonas Schöpf, 2024
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the type and functions for identifiers of function symbols.
-}
module Data.LCTRS.FIdentifier (
  -- * Core FId Functions
  FId (..),
  getFIdText,
  getFIdInSort,
  getFIdSort,
  updateFIdSort,
  fId,
  getArity,
  fIdParse,
) where

import Data.LCTRS.Sort (Sort, SortAnnotation, Sorted (..), inSort, outSort, sortAnnotation)
import Data.SMT (Theory (..), boolSort)
import Data.String (IsString)
import Prettyprinter (Pretty (..))

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

-- | Defines a data type for function symbol identifiers with special symbols for some logic symbols.
data FId f
  = FId f SortAnnotation
  | CEquation SortAnnotation
  | Eq SortAnnotation
  | Conj
  | Top
  | Neg
  | Imp
  | Disj
  | GrT SortAnnotation
  | FIdDp f SortAnnotation
  deriving (Eq, Ord)

-- | 'fIdParse' has the same behavior as 'fId' but wraps the correct identifier around special logic symbols.
fIdParse :: (IsString s, Eq s) => [Theory] -> s -> SortAnnotation -> FId s
fIdParse (_ : _) "=" sortann = Eq sortann
fIdParse (_ : _) "and" _ = Conj
fIdParse (_ : _) "true" _ = Top
fIdParse (_ : _) "not" _ = Neg
fIdParse (_ : _) "=>" _ = Imp
fIdParse (_ : _) "or" _ = Disj
fIdParse theories ">" sortann | Ints `elem` theories || Reals `elem` theories = GrT sortann
fIdParse theories "bvugt" sortann | FixedSizeBitVectors `elem` theories = GrT sortann
fIdParse _ name sortann = fId name sortann

instance (Pretty f) => Pretty (FId f) where
  pretty (FId f _) = pretty f
  pretty (CEquation _) = "≈"
  pretty (Eq _) = "="
  pretty Conj = "and"
  pretty Top = "true"
  pretty Neg = "not"
  pretty Imp = "=>"
  pretty Disj = "or"
  pretty (GrT _) = ">"
  pretty (FIdDp f _) = pretty f <> "#"

instance Sorted (FId f) where
  sort = getFIdOutSort

----------------------------------------------------------------------------------------------------
-- core functionality
----------------------------------------------------------------------------------------------------

-- | 'fId' @n@ @sa@ @typ@ returns given a symbol name @n@ and a sort annotation @sa@ and a symbol type @typ@ a function symbol identifier.
fId :: f -> SortAnnotation -> FId f
fId = FId

-- | 'getArity' returns the arity of the given function symbol identifier.
getArity :: FId f -> Int
getArity = length . getFIdInSort

updateFIdSort :: FId f -> SortAnnotation -> FId f
updateFIdSort (FId id _) sa = fId id sa
updateFIdSort (Eq _) sa = Eq sa
updateFIdSort (CEquation _) sa = CEquation sa
updateFIdSort Conj _ = Conj
updateFIdSort Top _ = Top
updateFIdSort Neg _ = Neg
updateFIdSort Imp _ = Imp
updateFIdSort Disj _ = Disj
updateFIdSort (GrT _) sa = GrT sa
updateFIdSort (FIdDp id _) sa = FIdDp id sa

-- | 'getFIdText' returns the function symbol identifier if possible, but the identifier is restricted to being of the typeclass 'IsString'.
getFIdText :: (IsString s) => FId s -> s
getFIdText (Eq _) = "="
getFIdText (CEquation _) = "≈"
getFIdText Conj = "and"
getFIdText Top = "true"
getFIdText Neg = "not"
getFIdText Imp = "=>"
getFIdText (FId id _) = id
getFIdText Disj = "or"
getFIdText (GrT _) = ">"
getFIdText (FIdDp id _) = id

-- | 'getFIdInSort' returns the input sorts of the given function symbol identifier.
getFIdInSort :: FId f -> [Sort]
getFIdInSort (Eq s) = inSort s
getFIdInSort (CEquation s) = inSort s
getFIdInSort Conj = [boolSort, boolSort]
getFIdInSort Top = []
getFIdInSort Neg = [boolSort]
getFIdInSort Imp = [boolSort, boolSort]
getFIdInSort (FId _ s) = inSort s
getFIdInSort Disj = [boolSort, boolSort]
getFIdInSort (GrT s) = inSort s
getFIdInSort (FIdDp _ s) = inSort s

-- | 'getFIdOutSort' returns the output sort of the given function symbol identifier.
getFIdOutSort :: FId f -> Sort
getFIdOutSort (Eq s) = outSort s
getFIdOutSort (CEquation s) = outSort s
getFIdOutSort Conj = boolSort
getFIdOutSort Top = boolSort
getFIdOutSort Neg = boolSort
getFIdOutSort Imp = boolSort
getFIdOutSort (FId _ s) = outSort s
getFIdOutSort Disj = boolSort
getFIdOutSort (GrT s) = outSort s
getFIdOutSort (FIdDp _ s) = outSort s

-- | 'getFIdSort' returns the sort annotation of the given function symbol identifier.
getFIdSort :: FId f -> SortAnnotation
getFIdSort (Eq s) = s
getFIdSort (CEquation s) = s
getFIdSort Conj = sortAnnotation [boolSort, boolSort] boolSort
getFIdSort Top = sortAnnotation [] boolSort
getFIdSort Neg = sortAnnotation [boolSort] boolSort
getFIdSort Imp = sortAnnotation [boolSort, boolSort] boolSort
getFIdSort (FId _ s) = s
getFIdSort Disj = sortAnnotation [boolSort, boolSort] boolSort
getFIdSort (GrT s) = s
getFIdSort (FIdDp _ s) = s
