{- |
Module      : Guard
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the implementation of Guards used for constrained rules.
-}
module Data.LCTRS.Guard (
  Guard,
  tryEvalGuard,
  collapseGuardToTerm,
  createGuard,
  modifyGuard,
  -- , setGuardValidBeCautious
  termsGuard,
  isTopGuard,
  varsGuard,
  guardVariants,
  funsGuard,
  conjGuards,
  disjGuards,
  mapGuard,
  mapGuardM,
  concatGuards,
  prettyGuard,
  prettyGuardFId,
  fromTerm,
) where

----------------------------------------------------------------------------------------------------
-- imports
----------------------------------------------------------------------------------------------------

import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.Term (Term)
import qualified Data.LCTRS.Term as T
import Data.List (sort)
import Prettyprinter
import qualified Rewriting.Substitution as Sub

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

newtype Guard f v = Guard [Term f v]
  deriving (Eq, Ord)

-- data Guard f v = Guard Bool [Term f v]
--   deriving (Eq, Ord)

----------------------------------------------------------------------------------------------------
-- core functions
----------------------------------------------------------------------------------------------------

createGuard :: [Term f v] -> Guard f v
-- createGuard ts = Guard False ts
createGuard = Guard

fromTerm :: Term f v -> Guard f v
fromTerm = createGuard . (: [])

modifyGuard :: Guard f v -> ([Term f v] -> [Term f' v']) -> Guard f' v'
modifyGuard (Guard ts) f = createGuard $ f ts

termsGuard :: Guard f v -> [Term f v]
termsGuard (Guard ts) = ts

-- setGuardValidBeCautious :: Guard f v -> Guard f v
-- setGuardValidBeCautious (Guard ts)  = Guard True ts

isTopGuard :: Guard f v -> Bool
-- isTopGuard (Guard True []) = True
isTopGuard (Guard []) = True
isTopGuard _ = False

funsGuard :: Guard f v -> [f]
funsGuard (Guard ts) = concatMap T.funs ts

varsGuard :: Guard f v -> [v]
varsGuard (Guard ts) = concatMap T.vars ts

conjGuards :: Guard f v -> Guard f v -> Guard f v
conjGuards (Guard ts) (Guard ts') = createGuard $ ts ++ ts'

disjGuards :: (Eq v, Eq f) => Guard (FId f) v -> Guard (FId f) v -> Guard (FId f) v
disjGuards g1 g2 = createGuard $ (: []) $ T.disj (collapseGuardToTerm g1) (collapseGuardToTerm g2)

concatGuards :: [Guard f v] -> Guard f v
concatGuards = foldr conjGuards (createGuard [])

mapGuard :: (Term f v -> Term f' v') -> Guard f v -> Guard f' v'
mapGuard f (Guard ts) = createGuard $ map f ts

mapGuardM :: (Monad m) => (Term f v -> m (Term f' v')) -> Guard f v -> m (Guard f' v')
mapGuardM f (Guard ts) = createGuard <$> mapM f ts

-- conjGuards :: [Term f v] -> Maybe (Term f v) -> Maybe (Term f v)
-- conjGuards Nothing Nothing = Nothing
-- conjGuards Nothing g = g
-- conjGuards g Nothing = g
-- conjGuards (Just g1) (Just g2) =

guardVariants :: (Eq f, Ord v, Ord f) => Guard f v -> Guard f v -> Bool
guardVariants (Guard ts) (Guard ts') = go (sort ts) (sort ts')
 where
  go [] [] = True
  go (x : xs) (y : ys) | Sub.isVariantOf x y = go xs ys
  go _ _ = False

collapseGuardToTerm :: (Eq v, Eq f) => Guard (FId f) v -> Term (FId f) v
collapseGuardToTerm (Guard ts) = foldl T.conj T.top ts

-- cannot deduce false at the moment
tryEvalGuard :: (Eq f, Eq v) => Guard (FId f) v -> Maybe Bool
-- tryEvalGuard (Guard True _) = return True
tryEvalGuard (Guard ts) = iter ts
 where
  iter [] = Just True
  iter [t]
    | t == T.top = Just True
    | otherwise = Nothing
  iter ts = go $ map (iter . (: [])) ts
   where
    go [] = Just True
    go (Nothing : _) = Nothing
    go (Just True : rest) = go rest
    go (Just False : _) = Just False

----------------------------------------------------------------------------------------------------
-- pretty printing
----------------------------------------------------------------------------------------------------

prettyGuard :: (Pretty f, Pretty v) => Guard f v -> Doc ann
prettyGuard guard -- (termsGuard -> terms)
  | isTopGuard guard = ""
  | otherwise = case termsGuard guard of
      [] -> brackets "true"
      [t] -> brackets $ pretty t
      ts -> brackets $ hsep $ punctuate comma $ map pretty ts

prettyGuardFId :: (Pretty f, Pretty v) => Guard (FId f) v -> Doc ann
prettyGuardFId guard -- (termsGuard -> terms)
  | isTopGuard guard = ""
  | otherwise = case termsGuard guard of
      [] -> brackets "true"
      [t] -> brackets $ T.prettyTermFId t
      ts -> brackets $ hsep $ punctuate comma $ map T.prettyTermFId ts
