{- |
Module      : Position
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides functions to create and modify position of
terms.
-}
module Data.LCTRS.Position (
  -- * Types and Typeclasses
  Pos,
  pos,

  -- * manipulation functions
  above,
  below,
  parallelTo,
  epsilon,
  position,
  append,
  prepend,
) where

----------------------------------------------------------------------------------------------------
-- imports
----------------------------------------------------------------------------------------------------

import Data.List (isPrefixOf)
import Prettyprinter (Pretty (pretty))
import Utils (dropCommonPrefix)

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

-- | 'Pos' specifies positions
newtype Pos = Pos
  { pos :: [Int]
  }

instance Eq Pos where
  (==) (Pos p) (Pos p') = p == p'

instance Ord Pos where
  compare (Pos p) (Pos p') = compare p p'

-- NOTE: this would give a severe bug with isSubsetOf of Data.Set
-- compare p p'
--   | above p p' = GT
--   | below p p' = LT
--   | p == p' = EQ
--   | otherwise = compare (pos p) (pos p') -- do we want this behaviour?

instance Pretty Pos where
  pretty (Pos []) = "ε"
  pretty (Pos xs) = foldr ((<>) . pretty) "" xs

instance Semigroup Pos where
  (Pos p) <> (Pos p') = position $ p <> p'

instance Monoid Pos where
  mempty = epsilon

instance Show Pos where
  show (Pos digits) = show digits

----------------------------------------------------------------------------------------------------
-- main funtionality
----------------------------------------------------------------------------------------------------

-- | smart constructor for Pos
position :: [Int] -> Pos
position = Pos

-- | 'above' @p@ @q@ returns 'Data.Boolean.true' if @p@ is above @q@ and 'Data.Boolean.false' otherwise.
above :: Pos -> Pos -> Bool
above (Pos p) (Pos q) = p `isPrefixOf` q

-- | 'below' @p@ @q@ returns 'Data.Boolean.true' if @p@ is below @q@ and 'Data.Boolean.false' otherwise.
below :: Pos -> Pos -> Bool
below = flip above

-- | 'parallelTo' @p@ @q@ returns 'Data.Boolean.true' if @p@ is parallel to @q@ and 'Data.Boolean.false' otherwise.
parallelTo :: Pos -> Pos -> Bool
parallelTo (Pos p) (Pos q) = not (null p') && not (null q')
 where
  (p', q') = dropCommonPrefix p q

-- | the root position epsilon.
epsilon :: Pos
epsilon = position []

-- | 'append' @p@ @i@ appends the integer @i@ at the end of position @p@.
append :: Pos -> Int -> Pos
append (Pos xs) i = position $ xs ++ [i]

-- | 'append' @p@ @i@ appends the integer @i@ at the end of position @p@.
prepend :: Int -> Pos -> Pos
prepend i (Pos xs) = position $ i : xs
