{- |

Module      : Sort
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides functions to create and modify sorts of
variables, function symbols and terms.
-}
module Data.LCTRS.Sort (
  -- * Type and Typeclasses
  Sort (..),
  Attr (..),
  Sorted (..),

  -- * Function on Sorts
  litSort,
  psort,
  attrsort,
  dpSort,
  sortIdentifier,
  attrIdentifier,
  isPolySort,

  -- * Types and Functions for Sort Annotations
  SortAnnotation,
  inSort,
  outSort,
  sortAnnotation,
  sortAnnotationFrom,
  containsPolySort,
  prettySortAnnotation,

  -- * Instance Checking for Sorts/Sort Annotations
  isInstanceSortAnnOf,
  isInstanceSortOf,
) where

----------------------------------------------------------------------------------------------------
-- imports
----------------------------------------------------------------------------------------------------

import Control.Monad.State.Strict (State, evalState, gets, modify)
import Data.Foldable (foldrM)
import qualified Data.Map.Strict as M
import Data.Text (Text, pack)
import Fmt (Buildable (build))
import Prettyprinter (Doc, Pretty, comma, hsep, parens, pretty, punctuate, (<+>))

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

-- | 'Sort' type defining literal and polymorphic sorts.
data Sort = LitSort Text | AttrSort Text Attr | PSort Text
  deriving (Eq, Ord)

data Attr = AttrInt Int | AttrPol Text
  deriving (Eq, Ord)

instance Pretty Attr where
  pretty (AttrInt i) = pretty i
  pretty (AttrPol t) = pretty t

instance Pretty Sort where
  pretty (AttrSort t i) = pretty t <> " " <> pretty i
  pretty s = pretty $ sortIdentifier s

instance Buildable Sort where
  build = build . show . pretty

----------------------------------------------------------------------------------------------------
-- Type Class to extract always sorts
----------------------------------------------------------------------------------------------------

class Sorted a where
  sort :: a -> Sort

----------------------------------------------------------------------------------------------------
-- functions on sorts
----------------------------------------------------------------------------------------------------

-- | 'sort' @text@ returns a literal sort of name @text@.
litSort :: Text -> Sort
litSort = LitSort

-- | 'psort' @text@ returns a polymorphic sort of name @text@.
psort :: Text -> Sort
psort = PSort

-- | 'attrsort' @text attr@ returns a sort of name @text@ with attribute @attr@.
attrsort :: Text -> Attr -> Sort
attrsort = AttrSort

-- | 'dpSort' defines the specific sort needed for DP symbols.
dpSort :: Sort
dpSort = litSort "dpSort"

-- | 'isPolySort' @sort@ checks if a given sort is a polymorphic sort.
isPolySort :: Sort -> Bool
isPolySort (LitSort _) = False
isPolySort (PSort _) = True
isPolySort (AttrSort _ (AttrPol _)) = True
isPolySort (AttrSort _ (AttrInt _)) = False

-- | 'sortIdentifier' @sort@ returns the attached name of a sort.
sortIdentifier :: Sort -> Text
sortIdentifier (LitSort t) = t
sortIdentifier (PSort t) = t
sortIdentifier (AttrSort t a) = t <> " " <> attrIdentifier a

attrIdentifier :: Attr -> Text
attrIdentifier (AttrInt i) = pack $ show i
attrIdentifier (AttrPol t) = t

instance Show Sort where
  show (LitSort s) = show s
  show (PSort s) = show s
  show (AttrSort t a) = show t <> " " <> show (pretty a)

----------------------------------------------------------------------------------------------------
-- sort annotation
----------------------------------------------------------------------------------------------------

-- | Abstract sort annotation with sorts of type @s@.
data ParSortAnnotation s = ParSortAnnotation
  { inSort :: [s]
  -- ^ 'inSort' returns the list of input sorts
  , outSort :: s
  -- ^ 'outSort' returns the output sort
  }
  deriving (Eq, Ord)

instance Functor ParSortAnnotation where
  fmap f sa = ParSortAnnotation (f <$> inSort sa) (f $ outSort sa)

-- | 'SortAnnotation' is the instantiated 'ParSortAnnotation' with type 'Sort'
type SortAnnotation = ParSortAnnotation Sort

-- | 'sortAnnotation' @ins@ @out@ returns a sort annotation with input sorts @in@ and output sort @out@.
sortAnnotation :: [Sort] -> Sort -> SortAnnotation
sortAnnotation = ParSortAnnotation

-- | 'sortAnnotationFrom' @f@ @ins@ @out@ returns a sort annotation with input sorts @in@ and output sort @out@ with the sort constructed by @f@.
sortAnnotationFrom :: (Text -> Sort) -> [Text] -> Text -> SortAnnotation
sortAnnotationFrom f ins out = ParSortAnnotation (map f ins) (f out)

-- | 'containsPolySort' @sa@ checks that a given sort annotation @sa@returns a sort annotation contains a polymorphic sort.with no input sorts and output sort 'boolSort'. This can be useful in type inference.
containsPolySort :: SortAnnotation -> Bool
containsPolySort sa = any isPolySort (inSort sa) || isPolySort (outSort sa)

-- | 'prettySortAnnotation' @sa@ pretty prints the sort annotation @sa@.
prettySortAnnotation :: SortAnnotation -> Doc ann
prettySortAnnotation sortAnnotation
  | null (inSort sortAnnotation) = pretty $ outSort sortAnnotation
  | length (inSort sortAnnotation) == 1 =
      pretty (head $ inSort sortAnnotation)
        <+> "->"
        <+> pretty (outSort sortAnnotation)
  | otherwise =
      parens (hsep $ punctuate comma (map pretty $ inSort sortAnnotation))
        <+> "->"
        <+> pretty (outSort sortAnnotation)

----------------------------------------------------------------------------------------------------
-- instance checking
----------------------------------------------------------------------------------------------------

-- | 'isInstanceSortOf' @s1@ @s2@ checks whether @s1@ is an instance of @s2@.
isInstanceSortOf :: Sort -> Sort -> Bool
isInstanceSortOf (LitSort s) (LitSort s') = s == s'
isInstanceSortOf _ (LitSort _) = False
isInstanceSortOf (AttrSort t _) (AttrSort t' (AttrPol _)) = t == t'
isInstanceSortOf (AttrSort t (AttrInt i)) (AttrSort t' (AttrInt i')) = t == t' && i == i'
isInstanceSortOf _ (AttrSort _ _) = False
isInstanceSortOf _ (PSort _) = True

-- | 'isInstanceSortAnnOf' @sa1@ @sa2@ checks whether @sa1@ is an instance of @sa2@.
isInstanceSortAnnOf :: SortAnnotation -> SortAnnotation -> Bool
isInstanceSortAnnOf sa1 sa2 | length (inSort sa1) /= length (inSort sa2) = False
isInstanceSortAnnOf sa1 sa2 = evalState combinations M.empty
 where
  combinations = foldrM go True $ zip (outSort sa1 : inSort sa1) (outSort sa2 : inSort sa2)

  go _ False = return False
  go (s1, s2) True = s1 `isInstance` s2

  isInstance :: Sort -> Sort -> State (M.Map Sort Sort) Bool
  isInstance instanceSort generalSort
    | instanceSort `isInstanceSortOf` generalSort = do
        cached <- gets (M.lookup generalSort)
        case cached of
          Just mappedSort -> return $ instanceSort == mappedSort
          Nothing -> do
            modify $ M.insert generalSort instanceSort
            return True
    | otherwise = return False
