{- |
Module      : Transformation
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides various transformation functions on LCTRSs without altering
its semantic.
-}
module Data.LCTRS.Transformation (
  moveValuesToGuard,
  moveLogicTermsOverLVarsToGuard,
  unifyRules,
) where

import Control.Monad (mapAndUnzipM)
import Data.Containers.ListUtils (nubOrd)
import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.Guard
import Data.LCTRS.LCTRS (LCTRS (rules), getRules, updateRules)
import qualified Data.LCTRS.Rule as R
import Data.LCTRS.Sort (Sorted (sort), sortAnnotation)
import qualified Data.LCTRS.Term as T
import Data.LCTRS.VIdentifier (VId, freshV)
import qualified Data.Map.Strict as M
import Data.Monad (FreshM, freshI)
import Data.SMT (boolSort)
import qualified Data.Set as S

moveValuesToGuard
  :: (Ord v, Ord f)
  => LCTRS (FId f) (VId v)
  -> FreshM (LCTRS (FId f) (VId v))
moveValuesToGuard lc = do
  updateRules lc <$> mapM go (getRules lc)
 where
  go rule = do
    let l = R.lhs rule
    let g = R.guard rule
    (l', newConsL) <- goTerm l
    let g' = createGuard $ newConsL ++ termsGuard g
    -- g'' <- checkValidityGuard g'
    return $ R.createRule l' (R.rhs rule) g'

  goTerm v@(T.Var _) = return (v, [])
  goTerm t@(T.Val f) = do
    i <- freshI
    let
      so = sort f
      v = T.var $ freshV so i
      sa = sortAnnotation [so, so] boolSort
    return (v, [T.eq sa v t])
  goTerm (T.Fun typ f args) = do
    (args', phis) <- mapAndUnzipM goTerm args
    return (T.fun typ f args', concat phis)

moveLogicTermsOverLVarsToGuard
  :: (Ord v, Ord f)
  => LCTRS (FId f) (VId v)
  -> FreshM (LCTRS (FId f) (VId v))
moveLogicTermsOverLVarsToGuard lc = do
  updateRules lc <$> mapM go (getRules lc)
 where
  go rule = do
    let l = R.lhs rule
    let g = R.guard rule
    (l', newConsL) <- goTerm (R.lvar rule) l
    let g' = createGuard $ newConsL ++ termsGuard g
    -- g'' <- checkValidityGuard g'
    return $ R.createRule l' (R.rhs rule) g'

  isLogicTermOverLVars lvars term = T.isLogicTerm term && S.fromList (T.vars term) `S.isSubsetOf` lvars

  goTerm lvars lterm | isLogicTermOverLVars lvars lterm = do
    i <- freshI
    let
      so = sort lterm
      v = T.var $ freshV so i
      sa = sortAnnotation [so, so] boolSort
    return (v, [T.eq sa v lterm])
  goTerm _ v@(T.Var _) = return (v, [])
  goTerm lvars (T.Fun typ f args) = do
    (args', phis) <- mapAndUnzipM (goTerm lvars) args
    return (T.fun typ f args', concat phis)

-- | unify rules for which lhss and rhss are variants
unifyRules
  :: (Ord v, Ord f)
  => LCTRS (FId f) (VId v)
  -> FreshM (LCTRS (FId f) (VId v))
unifyRules lctrs = do
  -- rename them back to have distinct variables for distinct rules
  renamedAgain <- mapM (R.renameFresh . R.rename snd) newRules
  -- renamedAgain <- newRules
  -- update LCTRS with those rules
  return $ lctrs{rules = renamedAgain}
 where
  -- old rules in LCTRS
  rules = getRules lctrs
  -- create unique Map for variables of each rule mapping to Int
  renamingMap =
    M.fromList
      [ (rule, M.fromList $ zip (nubOrd $ R.vars rule) [0 :: Int ..])
      | rule <- rules
      ]
  -- function to get correct integer for each variable wrt to a rule
  getV rule v = addType (R.lvar rule) v . freshV (sort v) $ (renamingMap M.! rule) M.! v
  -- distinguish between logical and non-logical (they should not rename each other)
  addType lvars v
    | v `S.member` lvars = (T.TheoryVar,)
    | otherwise = (T.TermVar,)
  -- rename function using the Map above
  uniqueRenamed =
    [ ((R.lhs renamed, R.rhs renamed), [R.guard renamed])
    | rule <- rules
    , let renamed = R.rename (getV rule) rule
    ]
  -- unify constraints of rules which are variants
  unified = M.fromListWith (++) uniqueRenamed
  -- put constraints into disjunction
  newRules = [R.createRule l r $ disjs gs | ((l, r), gs) <- M.toList unified]

  -- newRules = sequence [rule | ((l, r), gs) <- M.toList unified, let rule = shouldUnify l r gs]
  -- shouldUnify l r [] = R.renameFresh $ R.rename snd . R.createRule l r $ G.createGuard []
  -- shouldUnify _ _ [(rule,_)] = return rule
  -- shouldUnify l r gs = R.renameFresh $ R.rename snd . R.createRule l r $ disjs [g | (_, g) <- gs]

  disjs [] = createGuard []
  disjs gs@(_ : _) = foldr1 disjGuards gs

_splitRules
  :: LCTRS (FId f) (VId v)
  -> LCTRS (FId f) (VId v)
_splitRules = undefined
