{- |
Module      : VIdentifier
Description :
Copyright   : (c) Jonas Schöpf, 2024
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the type and functions for identifiers of variables.
-}
module Data.LCTRS.VIdentifier (
  -- * Core VId Functions
  VId (..),
  freshV,
  vId,
  getVId,
) where

import Data.LCTRS.Sort (Sort, Sorted, sort)
import Prettyprinter (Pretty (..))

-- | Defines a data type for standard variables and fresh constructed ones including their sort.
data VId v = VId v Sort | Fresh Int Sort
  deriving (Eq, Ord)

instance (Pretty v) => Pretty (VId v) where
  pretty (VId v _) = "!" <> pretty v
  pretty (Fresh i _) = "?" <> pretty i

instance Sorted (VId v) where
  sort = getVIdSort

----------------------------------------------------------------------------------------------------
-- core functionality
----------------------------------------------------------------------------------------------------

-- | 'getVId' returns either the chosen identifier for a variable or the unique number given to a fresh variable.
getVId :: VId v -> Either v Int
getVId (VId v _) = Left v
getVId (Fresh i _) = Right i

-- | 'getVIdSort' returns the assigned 'Sort' of a variable.
getVIdSort :: VId v -> Sort
getVIdSort (VId _ s) = s
getVIdSort (Fresh _ s) = s

-- | 'freshV' @s@ @i@ returns given a sort @s@ and an integer @i@ a fresh variable identifier.
freshV :: Sort -> Int -> VId v
freshV s i = Fresh i s

-- | 'vId' @n@ @s@ returns given a variable name @n@ and a sort @s@ a variable identifier.
vId :: v -> Sort -> VId v
vId = VId
