{- |
Module      : SExpr
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides the typeclass SExpr and its function toSExpr transforming a
respective type into an SExpr.
-}
module Data.SExpr where

import Data.LCTRS.FIdentifier (FId (..))
import Data.LCTRS.Sort (Attr (..), Sort (..), Sorted (sort), inSort, isPolySort)
import Data.LCTRS.Term (Term (..))
import Data.LCTRS.VIdentifier (VId (..))
import Data.SMT (Identifier (Identifier), boolSort, extractPosPartOfNegInt, isBVSort)
import qualified Data.Set as S
import Data.Text (unpack)
import Fmt (fmt, (+|), (|+))
import Prettyprinter (Pretty (pretty))
import SimpleSMT (SExpr (..))
import qualified SimpleSMT as SMT

class ToSExpr a where
  toSExpr :: a -> SExpr

instance ToSExpr Sort where
  toSExpr (AttrSort t (AttrInt i)) = SMT.const $ fmt $ "(_ " +| t |+ " " +| i |+ ")"
  toSExpr s
    | isPolySort s = error $ "No SExpr for polysort " ++ show (pretty s)
    | otherwise = SMT.const $ show $ pretty s

instance ToSExpr Identifier where
  toSExpr (Identifier t) =
    case extractPosPartOfNegInt t of
      Just i -> SMT.app (SMT.const "-") [SMT.const (show i)]
      Nothing -> SMT.const $ unpack t

instance (ToSExpr f) => ToSExpr (FId f) where
  toSExpr (FId f _) = toSExpr f -- SMT.const . show $ pretty f
  toSExpr (Eq _) = SMT.const "="
  toSExpr (CEquation _) = SMT.const "≈"
  toSExpr Neg = SMT.const "not"
  toSExpr Conj = SMT.const "and"
  toSExpr Top = SMT.const "true"
  toSExpr Imp = SMT.const "=>"
  toSExpr Disj = SMT.const "or"
  toSExpr (GrT sa) | boolSort == head (inSort sa) = undefined
  toSExpr (GrT sa) | isBVSort (head $ inSort sa) = SMT.const "bvugt"
  toSExpr (GrT _) = SMT.const ">"
  toSExpr (FIdDp f sa) = modifyAtom ("?dp_" <>) $ toSExpr (FId f sa) -- SMT.const . show $ "?dp_" <> pretty f

instance (ToSExpr v) => ToSExpr (VId v) where
  -- NOTE: we escape variable names in order to ensure
  -- compatibility with the SMTLIB identifiers
  -- toSExpr v@(VId _ _) = SMT.const $ show ("|" <> pretty v <> "|")
  -- toSExpr f@(Fresh _ _) = SMT.const $ show ("|" <> pretty f <> "|")
  toSExpr (VId v _) = toSExpr v
  toSExpr (Fresh f _) = SMT.const $ show ("|" <> pretty f <> "|")

instance (ToSExpr f, ToSExpr v) => ToSExpr (Term f v) where
  toSExpr (Var v) = toSExpr v
  toSExpr (Fun _ f []) = toSExpr f
  toSExpr (Fun _ f args) = SMT.app (toSExpr f) (map toSExpr args)

modifyAtom :: (String -> String) -> SExpr -> SExpr
modifyAtom f (Atom string) = Atom $ f string
modifyAtom f (List sexprs) = List $ map (modifyAtom f) sexprs

existsSExprOfTerm :: (ToSExpr v, ToSExpr f, Sorted v) => S.Set v -> Term f v -> SMT.SExpr
existsSExprOfTerm valueVars term =
  ( SMT.List
      [ SMT.Atom "exists"
      , SMT.List (S.toList $ S.map (\v -> SMT.List [toSExpr v, toSExpr $ sort v]) valueVars)
      , toSExpr term
      ]
  )

existsSExpr :: (ToSExpr v, Sorted v) => S.Set v -> SMT.SExpr -> SMT.SExpr
existsSExpr valueVars sexpr
  | null valueVars = sexpr
  | otherwise =
      ( SMT.List
          [ SMT.Atom "exists"
          , SMT.List (S.toList $ S.map (\v -> SMT.List [toSExpr v, toSExpr $ sort v]) valueVars)
          , sexpr
          ]
      )

forallSExpr :: (ToSExpr v, Sorted v) => S.Set v -> SMT.SExpr -> SMT.SExpr
forallSExpr valueVars sexpr
  | null valueVars = sexpr
  | otherwise =
      ( SMT.List
          [ SMT.Atom "forall"
          , SMT.List (S.toList $ S.map (\v -> SMT.List [toSExpr v, toSExpr $ sort v]) valueVars)
          , sexpr
          ]
      )
