{- s|
Module      : SMT
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable

This module provides functions to obtain logic symbols from
theory files specified in the SMTLIB standard.
-}
module Data.SMT (
  getTheory,
  AriParser,
  Logic (..),
  Property (..),
  Theory (..),
  TheoryContent (..),
  Identifier (..),
  UnsortedFId,
  whitespace,
  symbol,
  numeral,
  integer,
  lexeme,
  parens,
  boolSort,
  intSort,
  bvSort,
  isBVSort,
  realSort,
  sortIdent,
  isAnyValue,
  isAnyTheorySort,
  extractPosPartOfNegInt,
) where

import Control.Monad.Combinators (empty, (<|>))
import Control.Monad.State.Strict (StateT)
import Data.Char (isHexDigit)
import Data.LCTRS.Sort (
  Attr (..),
  Sort (AttrSort),
  SortAnnotation,
  attrIdentifier,
  attrsort,
  inSort,
  litSort,
  outSort,
  psort,
  sortAnnotation,
 )
import Data.List (find)
import Data.Monad (FreshM, Logic (..))
import Data.String (IsString (..))
import Data.Text (Text, pack)
import qualified Data.Text as DT
import Data.Void (Void)
import Fmt (Buildable)
import Prettyprinter (Pretty (..))
import Text.Megaparsec (
  MonadParsec (eof, takeWhile1P, try),
  Parsec,
  ParsecT,
  between,
  chunk,
  runParser,
  (<?>),
 )
import Text.Megaparsec.Char (char, space1)
import qualified Text.Megaparsec.Char.Lexer as L

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

data Theory = Core | Ints | Reals | Reals_Ints | FixedSizeBitVectors
  deriving (Show, Eq, Ord, Bounded, Enum)

newtype Identifier = Identifier {fromIdentifier :: Text}
  deriving (Eq, Ord, Buildable)

instance Show Identifier where
  show (Identifier t) = show t

instance Pretty Identifier where
  pretty (Identifier t) = pretty t

instance IsString Identifier where
  fromString s = Identifier $ pack s

type UnsortedFId = (Identifier, SortAnnotation)

-- data UnsortedFId = UnsortedFId
--   { symId :: !Text
--   , sortAnn :: !SortAnnotation
--   , prop :: !(Maybe Property)
--   }

data Property = LAssoc | RAssoc | Chain | Pairw
  deriving (Show, Eq)

type Parser = Parsec Void Text
type AriParser s = ParsecT Void Text (StateT s FreshM)

data TheoryContent state = TheoryContent
  { logic :: !Logic
  , isTheorySort :: Sort -> Bool
  , valueSort :: Identifier -> Maybe Sort
  , isValue :: Identifier -> Bool
  , theorySymbols :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
  , specialSymbols :: [AriParser state (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
  }

instance Semigroup (TheoryContent s) where
  t1 <> t2 =
    TheoryContent
      (logic t1 <> logic t2)
      (\s -> isTheorySort t1 s || isTheorySort t2 s)
      (\val -> valueSort t1 val <|> valueSort t2 val)
      (\val -> isValue t1 val || isValue t2 val)
      (theorySymbols t1 ++ theorySymbols t2)
      (specialSymbols t1 ++ specialSymbols t2)

instance Monoid (TheoryContent s) where
  mempty =
    TheoryContent
      ALL
      (const False)
      (const Nothing)
      (const False)
      mempty
      mempty

----------------------------------------------------------------------------------------------------
-- basic parsing combinators
----------------------------------------------------------------------------------------------------

lineComment :: (Monad m) => ParsecT Void Text m ()
lineComment = L.skipLineComment ";"

whitespace :: (Monad m) => ParsecT Void Text m ()
whitespace = L.space space1 lineComment empty

lexeme :: (Monad m) => ParsecT Void Text m a -> ParsecT Void Text m a
lexeme = L.lexeme whitespace

symbol :: (Monad m) => Text -> ParsecT Void Text m Text
symbol = L.symbol whitespace

parens :: (Monad m) => ParsecT Void Text m a -> ParsecT Void Text m a
parens = between (symbol "(") (symbol ")")

numeral :: (Monad m) => ParsecT Void Text m Int
numeral = lexeme L.decimal

integer :: (Monad m) => ParsecT Void Text m Int
integer = lexeme (negate <$> (chunk "-" >> L.decimal)) <|> numeral

real :: (Monad m) => ParsecT Void Text m Double
real =
  lexeme (try (negate <$> (chunk "-" >> L.float)))
    <|> lexeme L.float

sortIdent :: (Monad m) => ParsecT Void Text m Text
sortIdent =
  lexeme (takeWhile1P Nothing (not . (`DT.elem` " \t\n\r():;")))
    <?> "wrong sort identifier"

----------------------------------------------------------------------------------------------------
-- special treatment of negative integers
----------------------------------------------------------------------------------------------------

extractPosPartOfNegInt :: Text -> Maybe Int
extractPosPartOfNegInt t = case runParser (integer <* eof) "" t of
  Left _ -> Nothing
  Right i
    | i < 0 -> Just $ negate i
    | otherwise -> Nothing

----------------------------------------------------------------------------------------------------
-- parsing relevant information of an SMT lib theory file
-- http://smtlib.cs.uiowa.edu/theories-Core.shtml
-- http://smtlib.cs.uiowa.edu/theories-Ints.shtml
-- http://smtlib.cs.uiowa.edu/theories-Reals.shtml
-- https://smtlib.cs.uiowa.edu/theories-Reals_Ints.shtml
-- https://smtlib.cs.uiowa.edu/theories-FixedSizeBitVectors.shtml
----------------------------------------------------------------------------------------------------

allTheories :: [Theory]
allTheories = [minBound .. maxBound]

isAnyTheorySort :: Sort -> Bool
isAnyTheorySort s = any (($ s) . isTheorySort . getTheory) allTheories

isAnyValue :: Identifier -> Bool
-- isAnyValue identifier = any ($ identifier) [isBoolValue, isIntValue, isRealValue, isBVValue]
isAnyValue identifier = any (($ identifier) . isValue . getTheory) allTheories

-- | 'boolSort' gives the default boolean sort.
boolSort :: Sort
boolSort = litSort "Bool"

-- -- | 'dummySortAnn' returns a sort annotation with no input sorts and output sort 'boolSort'. This can be useful in type inference.
-- dummySortAnn :: SortAnnotation
-- dummySortAnn = sortAnnotation [] boolSort

isBoolValue :: Identifier -> Bool
isBoolValue (Identifier t) =
  case runParser (pBool <* eof) "" t of
    Left _ -> False
    Right _ -> True
 where
  pBool = chunk "true" <|> chunk "false" :: Parser Text

-- | 'intSort' gives the default integer sort.
intSort :: Sort
intSort = litSort "Int"

isIntValue :: Identifier -> Bool
isIntValue (Identifier t) = case runParser (integer <* eof) "" t of
  Left _ -> False
  Right _ -> True

-- | 'realSort' gives the default real sort.
realSort :: Sort
realSort = litSort "Real"

isRealValue :: Identifier -> Bool
isRealValue (Identifier t) = case runParser (real <* eof) "" t of
  Left _ -> False
  Right _ -> True

-- | 'bvSort' gives the default bitvector sort.
bvSort :: Int -> Sort
bvSort i
  | i > 0 = attrsort "BitVec" (AttrInt i)
  | otherwise = error $ "bitvector of length " <> show i <> " is not valid."

bvSortPoly :: Text -> Sort
bvSortPoly t = attrsort "BitVec" (AttrPol t)

isBVValue :: Identifier -> Bool
isBVValue (Identifier t) = case runParser (pBVValue <* eof) "" t of
  Left _ -> False
  Right _ -> True
 where
  pBVValue = prefix *> (pBin <|> pHex <|> specialConst)

  prefix = chunk "#" :: Parser Text
  pBin = char 'b' *> L.binary
  pHex = char 'x' *> (L.hexadecimal :: Parser Int)
  specialConst = parens $ do
    _ <- chunk "_ bv"
    -- NOTE: consume no trailing white-space
    x <- L.decimal
    _ <- char ' '
    _ <- numeral
    return x :: Parser Int

isBVSort :: Sort -> Bool
isBVSort (AttrSort "BitVec" a) = case runParser (attr <* eof) "" (attrIdentifier a) of
  Left _ -> False
  Right _ -> True
 where
  attr = Left <$> try nat <|> Right <$> sortIdent
  nat = lexeme (L.decimal :: Parser Integer)
isBVSort _ = False

sortOfBVValue :: Identifier -> Maybe Sort
sortOfBVValue (Identifier t) = case runParser (pBVValue <* eof) "" t of
  Left _ -> Nothing
  Right so -> Just so
 where
  pBVValue = prefix *> (pBin <|> pHex <|> specialConst)

  prefix = chunk "#" :: Parser Text
  isBinDigit x = x == '0' || x == '1'
  pBin = do
    bin <- char 'b' *> takeWhile1P Nothing isBinDigit :: Parser Text
    return . bvSort $ DT.length bin
  pHex = do
    hex <- char 'x' *> takeWhile1P Nothing isHexDigit :: Parser Text
    return . bvSort . (* 4) $ DT.length hex
  specialConst = parens $ do
    _ <- chunk "_ bv"
    -- NOTE: consume no trailing white-space
    _ <- L.decimal :: Parser Int
    _ <- char ' '
    n <- numeral
    return $ bvSort n

--

getTheory :: Theory -> TheoryContent state
getTheory Core =
  TheoryContent
    QF_UF
    (== boolSort)
    (\val -> if isBoolValue val then Just boolSort else Nothing)
    isBoolValue
    coreSyms
    mempty
getTheory Ints =
  TheoryContent
    NIA
    (== intSort)
    (\val -> if isIntValue val then Just intSort else Nothing)
    isIntValue
    intSyms
    [specialDiv]
getTheory Reals =
  TheoryContent
    NRA
    (== realSort)
    (\val -> if isRealValue val then Just realSort else Nothing)
    isRealValue
    realSyms
    mempty
getTheory Reals_Ints =
  TheoryContent
    AUFNIRA
    (\s -> s == intSort || s == realSort)
    (\val -> snd <$> find (\(p, _) -> p val) [(isIntValue, intSort), (isRealValue, realSort)])
    (\val -> isIntValue val || isRealValue val)
    reals_intsSyms
    [specialDiv]
getTheory FixedSizeBitVectors =
  TheoryContent
    QF_BV
    isBVSort
    sortOfBVValue
    isBVValue
    bvSyms
    bvSpecialSyms

bvSpecialSyms :: [AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
bvSpecialSyms =
  [ specialExtract
  , specialRepeat
  , specialZeroExtend
  , specialSignExtend
  , specialRotateLeft
  , specialRotateRight
  , specialBvConst
  ]

coreSyms :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
coreSyms =
  [ ((Identifier "true", sortAnnotation [] boolSort), Nothing, const True)
  , ((Identifier "false", sortAnnotation [] boolSort), Nothing, const True)
  , ((Identifier "not", sortAnnotation [boolSort] boolSort), Nothing, const True)
  , ((Identifier "=>", sortAnnotation [boolSort, boolSort] boolSort), pure RAssoc, const True)
  , ((Identifier "and", sortAnnotation [boolSort, boolSort] boolSort), pure LAssoc, const True)
  , ((Identifier "or", sortAnnotation [boolSort, boolSort] boolSort), pure LAssoc, const True)
  , ((Identifier "xor", sortAnnotation [boolSort, boolSort] boolSort), pure LAssoc, const True)
  , ((Identifier "=", sortAnnotation [psort "A", psort "A"] boolSort), pure Chain, const True)
  , ((Identifier "distinct", sortAnnotation [psort "A", psort "A"] boolSort), pure Pairw, const True)
  ,
    ( (Identifier "ite", sortAnnotation [boolSort, psort "A", psort "A"] (psort "A"))
    , Nothing
    , const True
    )
  ]

intSyms :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
intSyms =
  [ ((Identifier "-", sortAnnotation [intSort] intSort), Nothing, const True)
  , ((Identifier "-", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "+", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "*", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "div", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "mod", sortAnnotation [intSort, intSort] intSort), Nothing, const True)
  , ((Identifier "abs", sortAnnotation [intSort] intSort), Nothing, const True)
  , ((Identifier "<=", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier "<", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier ">=", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier ">", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  ]

realSyms :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
realSyms =
  [ ((Identifier "-", sortAnnotation [realSort] realSort), Nothing, const True)
  , ((Identifier "-", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "+", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "*", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "/", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "<=", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier "<", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier ">=", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier ">", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  ]

reals_intsSyms :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
reals_intsSyms =
  [ ((Identifier "-", sortAnnotation [intSort] intSort), Nothing, const True)
  , ((Identifier "-", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "+", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "*", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "div", sortAnnotation [intSort, intSort] intSort), pure LAssoc, const True)
  , ((Identifier "mod", sortAnnotation [intSort, intSort] intSort), Nothing, const True)
  , ((Identifier "abs", sortAnnotation [intSort] intSort), Nothing, const True)
  , ((Identifier "<=", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier "<", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier ">=", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier ">", sortAnnotation [intSort, intSort] boolSort), pure Chain, const True)
  , ((Identifier "-", sortAnnotation [realSort] realSort), Nothing, const True)
  , ((Identifier "-", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "+", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "*", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "/", sortAnnotation [realSort, realSort] realSort), pure LAssoc, const True)
  , ((Identifier "<=", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier "<", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier ">=", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier ">", sortAnnotation [realSort, realSort] boolSort), pure Chain, const True)
  , ((Identifier "to_real", sortAnnotation [intSort] realSort), Nothing, const True)
  , ((Identifier "to_int", sortAnnotation [realSort] intSort), Nothing, const True)
  , ((Identifier "is_int", sortAnnotation [realSort] boolSort), Nothing, const True)
  ]

bvSyms :: [(UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
bvSyms =
  let
    sa1 = sortAnnotation [bvSortPoly "m"] (bvSortPoly "m")
    sa2 = sortAnnotation [bvSortPoly "m", bvSortPoly "m"] (bvSortPoly "m")
  in
    [
      ( (Identifier "concat", sortAnnotation [bvSortPoly "i", bvSortPoly "j"] (bvSortPoly "m"))
      , Nothing
      , check1
      )
    , ((Identifier "bvnot", sa1), Nothing, check2)
    , ((Identifier "bvneg", sa1), Nothing, check2)
    , ((Identifier "bvand", sa2), pure LAssoc, check2)
    , ((Identifier "bvor", sa2), pure LAssoc, check2)
    , ((Identifier "bvadd", sa2), pure LAssoc, check2)
    , ((Identifier "bvmul", sa2), pure LAssoc, check2)
    , ((Identifier "bvudiv", sa2), Nothing, check2)
    , ((Identifier "bvurem", sa2), Nothing, check2)
    , ((Identifier "bvshl", sa2), Nothing, check2)
    , ((Identifier "bvlshr", sa2), Nothing, check2)
    , ((Identifier "bvult", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort), Nothing, check3)
    , -- logic extension symbols
      -- the following symbols are only defined in an extension of the bitvector logic QF_BV
      ((Identifier "bvnand", sa2), Nothing, const True)
    , ((Identifier "bvnor", sa2), Nothing, const True)
    , ((Identifier "bvxor", sa2), Nothing, const True)
    , ((Identifier "bvxnor", sa2), Nothing, const True)
    ,
      ( (Identifier "bvcomp", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] (bvSort 1))
      , Nothing
      , const True
      )
    , ((Identifier "bvsub", sa2), Nothing, const True)
    , ((Identifier "bvdiv", sa2), Nothing, const True)
    , ((Identifier "bvsrem", sa2), Nothing, const True)
    , ((Identifier "bvsmod", sa2), Nothing, const True)
    , ((Identifier "bvashr", sa2), Nothing, const True)
    ,
      ( (Identifier "bvule", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvugt", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvuge", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvslt", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvsle", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvsgt", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ,
      ( (Identifier "bvsge", sortAnnotation [bvSortPoly "m", bvSortPoly "m"] boolSort)
      , Nothing
      , const True
      )
    ]

check1 :: SortAnnotation -> Bool
check1 sortAnn
  | length iS == 2 =
      let
        in1 = getBVInt $ iS !! 0
        in2 = getBVInt $ iS !! 1
        out = getBVInt oS
      in
        case (in1, in2, out) of
          (Just i, Just j, Just m) -> i > 0 && j > 0 && i + j == m
          _ -> False
  | otherwise = False
 where
  iS = inSort sortAnn
  oS = outSort sortAnn

getBVInt :: Sort -> Maybe Int
getBVInt (AttrSort n (AttrInt i)) | n == "BitVec" = Just i
getBVInt _ = Nothing

check2 :: SortAnnotation -> Bool
check2 sortAnn =
  let oS = mapM getBVInt $ outSort sortAnn : inSort sortAnn
  in  maybe False (all (> 0)) oS

check3 :: SortAnnotation -> Bool
check3 sortAnn
  | length iS == 2 = maybe False (all (> 0)) $ mapM getBVInt iS
  | otherwise = False
 where
  iS = inSort sortAnn

-- parsing functions for special function symbols with _
wrapSpecial :: Text -> Text -> Text
wrapSpecial t is = "(_ " <> t <> " " <> is <> ")"

extractSingleton :: [a] -> Maybe a
extractSingleton [x] = Just x
extractSingleton _ = Nothing

specialDiv :: AriParser state (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialDiv = parse
 where
  sa = sortAnnotation [intSort] boolSort
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "divisible"
    _ <- char ' '
    n <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show n
    if n > 0
      then return ((identifier, sa), Nothing, const True)
      else fail "cannot parse divisible."

specialExtract :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialExtract = parse
 where
  check i j sa =
    let
      inS = getBVInt =<< extractSingleton (inSort sa)
      outS = getBVInt $ outSort sa
    in
      case (inS, outS) of
        (Just m, Just n) -> m > i && n == i - j + 1
        _ -> False
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "n")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "extract"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    _ <- char ' '
    j <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack (show i) <> " " <> pack (show j)
    if j >= 0 && i > j
      then
        return
          ((identifier, sa), Nothing, check i j)
      else fail "cannot parse extract."

specialRepeat :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialRepeat = parse
 where
  check i sa =
    let
      inS = getBVInt =<< extractSingleton (inSort sa)
      outS = getBVInt $ outSort sa
    in
      case (inS, outS) of
        (Just m, Just im) -> m * i == im
        _ -> False
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "i*m")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "repeat"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show i
    if i >= 1
      then return ((identifier, sa), Nothing, check i)
      else fail "cannot parse repeat."

specialZeroExtend :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialZeroExtend = parse
 where
  check i sa =
    let
      inS = getBVInt =<< extractSingleton (inSort sa)
      outS = getBVInt $ outSort sa
    in
      case (inS, outS) of
        (Just m, Just im) -> m + i == im
        _ -> False
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "m+i")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "zero_extend"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show i
    if i >= 0
      then return ((identifier, sa), Nothing, check i)
      else fail "cannot parse zero_extend."

specialSignExtend :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialSignExtend = parse
 where
  check i sa =
    let
      inS = getBVInt =<< extractSingleton (inSort sa)
      outS = getBVInt $ outSort sa
    in
      case (inS, outS) of
        (Just m, Just im) -> m + i == im
        _ -> False
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "m+i")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "sign_extend"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show i
    if i >= 0
      then return ((identifier, sa), Nothing, check i)
      else fail "cannot parse sign_extend."

specialRotateLeft :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialRotateLeft = parse
 where
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "m")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "rotate_left"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show i
    if i >= 0
      then return ((identifier, sa), Nothing, const True)
      else fail "cannot parse rotate_left."

specialRotateRight :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialRotateRight = parse
 where
  sa = sortAnnotation [bvSortPoly "m"] (bvSortPoly "m")
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "rotate_right"
    _ <- char ' '
    i <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial id $ pack $ show i
    if i >= 0
      then return ((identifier, sa), Nothing, const True)
      else fail "cannot parse rotate_right."

specialBvConst :: AriParser s (UnsortedFId, Maybe Property, SortAnnotation -> Bool)
specialBvConst = parse
 where
  sa n = sortAnnotation [] (bvSort n)
  parse = parens $ do
    _ <- char '_'
    _ <- char ' '
    id <- chunk "bv"
    x <- L.decimal :: ParsecT Void Text m Int
    _ <- char ' '
    n <- L.decimal :: ParsecT Void Text m Int
    let identifier = Identifier $ wrapSpecial (id <> pack (show x)) (pack (show n))
    if n > 0
      then return ((identifier, sa n), Nothing, const True)
      else fail "cannot parse bv bitvector constant."
