{- |
Module      : Ari
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides functions the parser to parse LCTRSs in the ARI format.
It follows the specification at http://project-coco.uibk.ac.at/ARI/lctrs.php
-}
module Parser.Ari (
  fromFile,
  fromFile',
  fromFileFun,
  Input (..),
  DefaultInput,
  UnsortedFId,
  UnsortedVId,
  UnsortedTerm,
  UnsortedRule,
  getLCTRS,
  getFSorts,
  getThSorts,
  getVarSort,
  getValueCheck,
  getValueSort,
  getTransl,
  getFormat,
  getLogic,
  Format (..),
  pFormat,
) where

----------------------------------------------------------------------------------------------------
-- imports
----------------------------------------------------------------------------------------------------

import Control.Applicative.Combinators ((<|>))
import Control.Arrow (Arrow (..))
import Control.Monad (replicateM, when)
import Control.Monad.State.Strict (
  MonadIO (liftIO),
  MonadTrans (lift),
  StateT,
  get,
  gets,
  modify,
  runStateT,
 )
import Data.Char (isDigit, isLetter)
import Data.Containers.ListUtils (nubOrd)
import Data.Foldable (fold)
import Data.Functor (($>))
import Data.LCTRS.Guard (createGuard)
import Data.LCTRS.LCTRS (
  LCTRS,
  createLctrs,
  emptyLCTRS,
  getFuns,
  getRules,
  getTheoryFuns,
 )
import Data.LCTRS.Rule (createRule)
import qualified Data.LCTRS.Rule as R
import Data.LCTRS.Sort (
  Attr (AttrInt),
  Sort,
  SortAnnotation,
  attrsort,
  inSort,
  litSort,
  outSort,
  psort,
  sortAnnotation,
 )
import Data.LCTRS.Term (
  termFun,
  theoryFun,
  val,
  var,
 )
import qualified Data.LCTRS.Term as T
import Data.List (
  intersect,
 )
import qualified Data.Map as M
import Data.Maybe (
  maybeToList,
 )
import Data.Monad (FreshM, freshI)
import Data.SMT (
  Identifier (..),
  Logic,
  Property,
  Theory (..),
  TheoryContent,
  getTheory,
  isAnyValue,
  lexeme,
  numeral,
  parens,
  sortIdent,
  symbol,
  whitespace,
 )
import qualified Data.SMT as DSMT
import qualified Data.Set as S
import Data.Text (Text, pack)
import qualified Data.Text as DT
import Data.Void (Void)
import Fmt (Builder, fmt, nameF, (+|), (|+))
import Text.Megaparsec (
  MonadParsec (eof, takeWhile1P, try),
  ParseErrorBundle,
  ParsecT,
  errorBundlePretty,
  many,
  noneOf,
  optional,
  runParserT,
  (<?>),
 )
import Text.Megaparsec.Char (char)
import Utils (fst3)

----------------------------------------------------------------------------------------------------
-- types
----------------------------------------------------------------------------------------------------

data Format
  = F_TRS
  | F_CTRS
  | F_MSTRS
  | F_LCTRS
  | F_CSTRS
  | F_CSCTRS
  deriving (Eq)

instance Show Format where
  show F_TRS = "TRS"
  show F_CTRS = "CTRS"
  show F_MSTRS = "MSTRS"
  show F_LCTRS = "LCTRS"
  show F_CSTRS = "CSTRS"
  show F_CSCTRS = "CSCTRS"

type UnsortedFId = DSMT.UnsortedFId
type UnsortedVId = (Identifier, Int, Sort)

type UnsortedTerm = T.Term UnsortedFId UnsortedVId
type UnsortedRule = R.Rule UnsortedFId UnsortedVId

data Input f v = Input
  { lctrs :: !(LCTRS f v)
  , fSorts :: !(M.Map Identifier SortAnnotation)
  , thSorts :: M.Map Identifier [(SortAnnotation, Maybe Property, SortAnnotation -> Bool)]
  , varSort :: !(M.Map (R.Rule f v) (M.Map v Sort))
  , valueCheck :: f -> Bool
  , valueSort :: Identifier -> Maybe Sort
  , transl :: !(M.Map Identifier v)
  , format :: !(Maybe Format)
  , logic :: !Logic
  }

type DefaultInput = Input UnsortedFId UnsortedVId

emptyInput :: Input UnsortedFId UnsortedVId
emptyInput =
  Input
    emptyLCTRS
    M.empty
    M.empty
    M.empty
    (const False)
    (const Nothing)
    M.empty
    Nothing
    mempty

type Parser = DSMT.AriParser DefaultInput -- ParsecT Void Text (StateT DefaultInput FreshM)

----------------------------------------------------------------------------------------------------
-- state monad helper functions
----------------------------------------------------------------------------------------------------

addSpecialThSort -- FIXME: avoid multiple additions of values
  :: (Identifier, (SortAnnotation, Maybe Property, SortAnnotation -> Bool))
  -> StateT DefaultInput FreshM ()
addSpecialThSort (k, v) = do
  inp <- get
  let old = thSorts inp
  modify $ \inp -> inp{thSorts = M.insertWith (++) k [v] old}

setLCTRS
  :: LCTRS UnsortedFId UnsortedVId -> Parser ()
setLCTRS lc = modify $ \inp -> (inp{lctrs = lc})

getLCTRS
  :: Input f v -> LCTRS f v
getLCTRS = lctrs

setFSorts :: M.Map Identifier SortAnnotation -> Parser ()
setFSorts m = modify $ \inp -> inp{fSorts = m}

getFSorts :: Input f v -> M.Map Identifier SortAnnotation
getFSorts = fSorts

setThSorts
  :: M.Map Identifier [(SortAnnotation, Maybe Property, SortAnnotation -> Bool)] -> Parser ()
setThSorts m = modify $ \inp -> inp{thSorts = m}

getThSorts
  :: Input f v -> M.Map Identifier [(SortAnnotation, Maybe Property, SortAnnotation -> Bool)]
getThSorts = thSorts

setVarSort :: M.Map UnsortedRule (M.Map UnsortedVId Sort) -> Parser ()
setVarSort v = modify $ \inp -> inp{varSort = v}

getVarSort :: Input f v -> M.Map (R.Rule f v) (M.Map v Sort)
getVarSort = varSort

setValueSort :: (Identifier -> Maybe Sort) -> Parser ()
setValueSort v = modify $ \inp -> inp{valueSort = v}

getValueSort :: Input f v -> (Identifier -> Maybe Sort)
getValueSort = valueSort

setValueCheck :: (UnsortedFId -> Bool) -> Parser ()
setValueCheck v = modify $ \inp -> inp{valueCheck = v}

getValueCheck :: Input f v -> (f -> Bool)
getValueCheck = valueCheck

updateTransl
  :: (M.Map Identifier UnsortedVId -> M.Map Identifier UnsortedVId) -> Parser ()
updateTransl f = do
  inp <- get
  let old = transl inp
  modify $ \inp -> (inp{transl = f old})

setFormat :: Maybe Format -> Parser ()
setFormat fmt = modify $ \inp -> inp{format = fmt}

getFormat :: Input f v -> Maybe Format
getFormat = format

getTransl :: Input f v -> M.Map Identifier v
getTransl = transl

getLogic :: Input f v -> Logic
getLogic = logic

freshVar :: Identifier -> Parser UnsortedVId
freshVar v = do
  trans <- gets transl
  case M.lookup v trans of
    Nothing -> do
      c <- lift $ lift freshI
      c' <- lift $ lift freshI
      let s = psort . ('?' `DT.cons`) $ (pack . show) c'
      updateTransl (M.insert v (v, c, s))
      return (v, c, s)
    Just vi -> return vi

----------------------------------------------------------------------------------------------------
-- main parsing functions
----------------------------------------------------------------------------------------------------

ariParser :: Parser ()
ariParser = whitespace *> pAri <* eof

fromFile :: FilePath -> FreshM (Either (ParseErrorBundle Text Void) (), DefaultInput)
fromFile fp = do
  inp <- pack <$> liftIO (readFile fp)
  flip runStateT emptyInput $ runParserT ariParser fp inp

fromFileFun :: FilePath -> Parser a -> FreshM (Either (ParseErrorBundle Text Void) a, DefaultInput)
fromFileFun fp f = do
  inp <- pack <$> liftIO (readFile fp)
  flip runStateT emptyInput $ runParserT f fp inp

fromFile' :: FilePath -> FreshM DefaultInput
fromFile' fp = do
  pM <- fromFile fp
  case pM of
    (Left e, _) -> error $ errorBundlePretty e
    (Right _, s) ->
      if null
        ( map fst (S.toList (getFuns (lctrs s) `S.union` getTheoryFuns (lctrs s)))
            `intersect` map fst3 (concatMap R.vars (getRules $ lctrs s))
        )
        then return s
        else fmtError "variables and function symbols share identifiers."

----------------------------------------------------------------------------------------------------
-- basic combinators
----------------------------------------------------------------------------------------------------

reservedAriKeywords :: [Text]
reservedAriKeywords =
  [ "meta-info"
  , "comment"
  , "format"
  , "fun"
  , "sort"
  , "theory"
  , "rule"
  , ":guard"
  , ":var"
  , "->"
  , "define-fun"
  , ":smtlib"
  -- , "forall"
  -- , "exists"
  ]

reservedSMTLIBKeywords :: [Text]
reservedSMTLIBKeywords =
  [ "BINARY"
  , "DECIMAL"
  , "HEXADECIMAL"
  , "NUMERAL"
  , "STRING"
  , "_"
  , "!"
  , "as"
  , "let"
  , "exists"
  , "forall"
  , "match"
  , "par"
  ]

-- symbolIdent :: (Monad m) => ParsecT Void Text m Identifier
-- symbolIdent = lexeme $ do
--   --
--   -- if iden `elem` keywords
--   --   then error $ "function symbol identifier is reserved keyword: " ++ iden
--   --   else return iden <?> "wrong function identifier"
--   --
--   -- takeWhile1P Nothing (not . (`DT.elem` " \t\n\r():;≈"))
--   --
--   iden <- takeWhile1P Nothing (\c -> c `DT.elem` "~!@$%^&*_-+=<>.?/" || isLetter c || isDigit c)
--   if iden `elem` reservedSMTLIBKeywords
--     then error $ "function symbol identifier is reserved keyword: " <> show iden
--     else return (Identifier iden) <?> "wrong function identifier"

-- variableIdent :: (Monad m) => ParsecT Void Text m Text
-- variableIdent = lexeme $ do
--   firstChar <- asciiChar
--   when (isDigit firstChar) (fail "variable is not allowed to start with a digit.")
--   suff <- takeWhile1P Nothing (not . (`DT.elem` " \t\n\r():;≈"))
--   let iden = DT.cons firstChar suff
--   when
--     (("#x" `DT.isInfixOf` iden) || ("#b" `DT.isInfixOf` iden)) -- FIXME: make more efficient
--     (fail "variable is not allowed to start with a digit.")
--   return iden

ariIdentifier :: (Monad m) => ParsecT Void Text m Identifier
ariIdentifier = lexeme $ do
  iden <- takeWhile1P Nothing (not . (`DT.elem` " \t\n\r():;≈")) <?> "ARI identifier"
  return (Identifier iden)

smtlibIdentifier :: (Monad m) => ParsecT Void Text m Identifier
smtlibIdentifier = lexeme $ do
  iden <- takeWhile1P Nothing (\c -> c `DT.elem` "~!@$%^&*_-+=<>.?/" || isLetter c || isDigit c)
  if iden `elem` reservedSMTLIBKeywords
    then fmtError $ "identifier " +| iden |+ " is a reserved keyword in SMTLIB."
    else return (Identifier iden) <?> "SMTLIB identifier (simple symbol)"

optAriKeyword :: Text -> Parser a -> Parser (Maybe a)
optAriKeyword s p = optional $ ariKeyword s *> try p

ariKeyword :: Text -> Parser ()
ariKeyword w = do
  s <- symbol w
  if s `elem` reservedAriKeywords
    then return ()
    else fmtError $ s |+ " is not a keyword."

----------------------------------------------------------------------------------------------------
-- ARI format parser
----------------------------------------------------------------------------------------------------

pAri :: Parser ()
pAri = do
  fmt <- pFormat
  setFormat $ Just fmt
  if fmt == F_LCTRS
    then pAriLCTRS
    else
      if fmt == F_TRS
        then pAriTRS
        else
          if fmt == F_MSTRS
            then pAriMSTRS
            else fmtError $ "format " +| show fmt |+ " of given problem not supported."

----------------------------------------------------------------------------------------------------
-- parser to parse an TRS in the ARI format
----------------------------------------------------------------------------------------------------

pAriTRS :: Parser ()
pAriTRS = do
  -- parse theories given and combine them
  setValueCheck (const False)
  setValueSort (const Nothing)
  setThSorts M.empty
  -- parse signature
  fs <- S.fromList <$> many (try pFun)
  setFSorts $ M.fromList $ S.toList fs
  -- parse rules
  rsvarL <- many $ pRule [] (== singleSort)
  setVarSort M.empty
  -- create LCTRS
  let lctrs =
        createLctrs
          []
          (const False)
          []
          S.empty
          (S.singleton singleSort)
          fs
          (map fst rsvarL)
  setLCTRS lctrs
  return ()
 where
  singleSort = litSort "Unit"

  pFun :: Parser UnsortedFId
  pFun = parens $ do
    tS <- gets thSorts
    ariKeyword "fun"
    i <- ariIdentifier
    s <- numeral
    if i `M.member` tS
      then fmtError $ "signature symbol " +| i |+ " is already theory symbol"
      else return (i, sortAnnotation (replicate s singleSort) singleSort)

----------------------------------------------------------------------------------------------------
-- parser to parse an MSTRS in the ARI format
----------------------------------------------------------------------------------------------------

pAriMSTRS :: Parser ()
pAriMSTRS = do
  -- parse theories given and combine them
  setValueCheck (const False)
  setValueSort (const Nothing)
  setThSorts M.empty
  -- parse custom sorts
  parsedSorts <- S.fromList <$> many (try pSortDeclaration)
  -- all sorts (including theory sorts)
  let isKnownSort s = s `S.member` parsedSorts
  -- parse signature
  fs <- S.fromList <$> many (try $ pFun isKnownSort)
  setFSorts $ M.fromList $ S.toList fs
  -- parse rules
  rsvarL <- many $ pRule [] isKnownSort
  let vs = foldr combineVars M.empty rsvarL
  setVarSort vs
  -- create LCTRS
  let lctrs =
        createLctrs
          []
          (const False)
          []
          S.empty
          parsedSorts
          fs
          (map fst rsvarL)
  setLCTRS lctrs
  return ()
 where
  combineVars (_, Nothing) m = m
  combineVars (r, Just m) m' = M.insert r m m'

----------------------------------------------------------------------------------------------------
-- parser to parse an LCTRS in the ARI format
----------------------------------------------------------------------------------------------------

pAriLCTRS :: Parser ()
pAriLCTRS = do
  -- parse theories given and combine them
  (theoryNames, theoryContents) <- do
    (n, c) <- pTheoryCmd <?> "Core, Ints, Reals, Reals_Ints or FixedSizeBitVectors"
    (ns, cs) <-
      unzip <$> many (try pTheoryCmd) <?> "Core, Ints, Reals, Reals_Ints or FixedSizeBitVectors"
    if Core `elem` (n : ns)
      then return (n : ns, c : cs)
      else return (Core : n : ns, getTheory Core : c : cs)
  let theoryContent = mconcat theoryContents
  let theorySyms = DSMT.theorySymbols theoryContent
  setValueCheck
    ( \(i, sa) ->
        DSMT.isValue theoryContent i
          && maybe False ((sa ==) . sortAnnotation []) (DSMT.valueSort theoryContent i)
    )
  setValueSort (DSMT.valueSort theoryContent)
  let thM =
        M.fromListWith (++) [(t, [(sa, prop, check)]) | ((t, sa), prop, check) <- theorySyms]
  setThSorts thM
  -- parse defines
  defines <- many (try $ pDefine (DSMT.isTheorySort theoryContent))
  let definedSymbols = map fst defines
  -- define symbol sort pairs
  let dM =
        M.fromListWith (++) [(id, [(sa, Nothing, const True)]) | (id, sa) <- definedSymbols]
  -- all theory symbols
  let allThSyms = M.unionWith (++) thM dM
  setThSorts allThSyms
  -- parse custom sorts
  parsedSorts <- S.fromList <$> many (try pSortDeclaration)
  -- all sorts (including theory sorts)
  let isKnownSort s = s `S.member` parsedSorts || DSMT.isTheorySort theoryContent s
  -- parse signature
  fs <- S.fromList <$> many (try $ pFun isKnownSort)
  setFSorts $ M.fromList $ S.toList fs
  -- parse rules
  rsvarL <-
    many $ pRule (DSMT.specialSymbols theoryContent) isKnownSort
  let vs = foldr combineVars M.empty rsvarL
  setVarSort vs
  -- create LCTRS
  let lctrs =
        createLctrs
          (nubOrd theoryNames)
          (DSMT.isTheorySort theoryContent)
          defines
          (S.fromList (map fst3 theorySyms) <> S.fromList definedSymbols)
          parsedSorts
          fs
          (map fst rsvarL)
  setLCTRS lctrs
  return ()
 where
  combineVars (_, Nothing) m = m
  combineVars (r, Just m) m' = M.insert r m m'

-- parser for format header

--  https://smtlib.cs.uiowa.edu/language.shtml
data SMTLIBVersion
  = SMTLIB_2_6 -- https://smtlib.cs.uiowa.edu/papers/smt-lib-reference-v2.6-r2021-05-12.pdf
  -- \| SMTLIB_3_0_proposal -- https://smtlib.cs.uiowa.edu/version3.shtml

pSMTLIB :: Parser SMTLIBVersion
-- pSMTLIB = (symbol "SMT-LIB_2_6" $> SMT_LIB_2_6) <|> (symbol "SMT-LIB_3_0_proposal" $> SMT_LIB_3_0_proposal)
pSMTLIB = symbol "2.6" $> SMTLIB_2_6

-- http://project-coco.uibk.ac.at/ARI/lctrs.php
-- NOTE: function is currently also used for the LCTRS tagging tool
pFormat :: Parser Format
pFormat = parens $ do
  ariKeyword "format"
  format <-
    ((symbol "TRS" $> F_TRS) <?> "TRS")
      <|> ((symbol "CTRS" $> F_CTRS) <?> "CTRS")
      <|> ((symbol "MSTRS" $> F_MSTRS) <?> "MSTRS")
      <|> ((symbol "LCTRS" $> F_LCTRS) <?> "LCTRS")
      <|> ((symbol "CSTRS" $> F_CSTRS) <?> "CSTRS")
      <|> ((symbol "CSCTRS" $> F_CSCTRS) <?> "CSCTRS")
  -- for now ignore the SMTLIB version
  _ <- optional $ optAriKeyword ":smtlib" pSMTLIB
  return format

-- parser for theory declarations
pTheoryCmd
  :: Parser
      ( Theory
      , TheoryContent DefaultInput
      )
pTheoryCmd = parens $ do
  ariKeyword "theory"
  l <-
    ( symbol "Core"
        $> Core
        <|> symbol "Ints"
          $> Ints
        <|> symbol "Reals_Ints"
          $> Reals_Ints
        <|> symbol "Reals"
          $> Reals
        <|> symbol "FixedSizeBitVectors"
          $> FixedSizeBitVectors
      )
  let t = getTheory l
  return (l, t)

-- parser for sort declarations
pSortDeclaration :: Parser Sort
pSortDeclaration = parens $ ariKeyword "sort" *> (litSort <$> sortIdent)

-- parser for newly defined symbols
pDefine
  :: (Sort -> Bool)
  -> Parser (UnsortedFId, (([(Identifier, Sort)], Sort), Text))
pDefine isKnownSort = do
  updateTransl (const M.empty) -- reset cached vars
  thNames <- M.keys . getThSorts <$> get
  parens $ do
    ariKeyword "define-fun"
    i <- smtlibIdentifier
    if i `elem` thNames
      then fmtError $ "Identifier " +| i |+ " of defined symbol is already defined in a theory."
      else do
        inSs <- parens $ pVarSortDeclarations isKnownSort
        when
          (null inSs)
          ( fmtError $
              "You cannot define a value with define-fun: " +| i |+ ""
          )
        outS <- pSort isKnownSort
        -- exp <- pTerm fs
        exp <- pBalancedSMTTerm
        -- exp <- pFormula specialSymParsers isKnownSort
        let defineAnn = (map (first (\(a, _, _) -> a)) inSs, outS)
        let f = (i,) $ sortAnnotation (map snd inSs) outS
        return (f, (defineAnn, exp))

-- parser for function symbols
pFun :: (Sort -> Bool) -> Parser UnsortedFId
pFun isKnownSort = parens $ do
  tS <- gets thSorts
  ariKeyword "fun"
  i <- ariIdentifier
  s <- pFunSortDeclaration isKnownSort
  if i `M.member` tS
    then fmtError $ "signature symbol " +| i |+ " is already theory symbol"
    else return (i, s)

pFunSortDeclaration :: (Sort -> Bool) -> Parser SortAnnotation
pFunSortDeclaration isKnownSort = do
  uncurry sortAnnotation <$> (try singleSort <|> fullSortDecl)
 where
  singleSort = ([],) <$> pSort isKnownSort

  fullSortDecl = parens $ do
    ariKeyword "->"
    sorts <- many $ pSort isKnownSort
    if length sorts < 2
      then fmtError "invalid sort declaration for function symbol."
      else return (init sorts, last sorts)

-- parser for rules
pRule
  :: [Parser (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
  -> (Sort -> Bool)
  -> Parser (UnsortedRule, Maybe (M.Map UnsortedVId Sort))
pRule specialSymParsers isKnownSort = parens $ do
  updateTransl (const M.empty) -- reset cached vars
  ariKeyword "rule"
  s <- pTerm specialSymParsers
  t <- pTerm specialSymParsers
  -- g <- optAriKeyword ":guard" (pFormula specialSymParsers isKnownSort)
  g <- optAriKeyword ":guard" $ pTerm specialSymParsers
  v <- optAriKeyword ":var" $ M.fromList <$> parens (pVarSortDeclarations isKnownSort)
  return (createRule s t (createGuard $ maybeToList g), v)

-- pFormula
--   :: [Parser (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
--   -> (Sort -> Bool)
--   -> Parser UnsortedTerm
-- pFormula specialSymParsers isKnownSort = try (pQuantifier pForall) <|> try (pQuantifier pExists) <|> pTerm specialSymParsers
--  where
--   pQuantifier pQuant = parens $ do
--     f <- pQuant
--     decls <- pVarSortDeclarations isKnownSort
--     formula <- pFormula specialSymParsers isKnownSort
--     return $ f decls formula
--   pForall = ariKeyword "forall" $> undefined
--   pExists = ariKeyword "exists" $> undefined

pTerm
  :: [Parser (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
  -> Parser UnsortedTerm
pTerm specialSymParsers =
  try (parens $ pNonConstant specialSymParsers pIdentifier)
    <|> (try (pConstant pIdentifier) <|> pVariable pIdentifier)
 where
  pIdentifier = try ariIdentifier <|> pSPIdent
   where
    pSPIdent = do
      ((i, s), property, check) <- case specialSymParsers of
        [] -> fail "no special symbol given by theory."
        ps -> foldr1 ((<|>) . try) ps
      _ <- lift $ addSpecialThSort (i, (s, property, check))
      return i

pNonConstant
  :: [Parser (UnsortedFId, Maybe Property, SortAnnotation -> Bool)]
  -> Parser Identifier
  -> Parser UnsortedTerm
pNonConstant specialSymParsers pIdentifier = do
  i <- pIdentifier
  args <- (:) <$> pTerm specialSymParsers <*> many (pTerm specialSymParsers)
  theorySymMap <- gets thSorts
  termSymMap <- gets fSorts
  sa <- getSortAnn i (length args)
  let f = (i, sa)
  if i `M.member` termSymMap
    then do
      return $ termFun f args
    else
      if i `M.member` theorySymMap
        then return $ theoryFun f args
        else fmtError $ " unknown function symbol " +| i |+ " ."

pConstant
  :: Parser Identifier
  -> Parser UnsortedTerm
pConstant pIdentifier = do
  inp <- get
  i <- pIdentifier
  fSyms <- gets fSorts
  case valueSort inp i of
    Just sort
      | valueCheck inp (i, sortAnnotation [] sort) ->
          -- value symbol
          return $ val (i, sortAnnotation [] sort)
    _ | i `M.member` fSyms -> do
      -- constant term symbol
      sa <- getSortAnn i 0
      return $ termFun (i, sa) []
    _ -> fail "value or constant term symbol"

pVariable
  :: Parser Identifier
  -> Parser UnsortedTerm
pVariable pIdentifier = do
  inp <- get
  i <- pIdentifier
  thSyms <- gets thSorts
  fSyms <- gets fSorts
  case valueSort inp i of
    Just sort
      | valueCheck inp (i, sortAnnotation [] sort)
          -- NOTE: do not allow variables named as function symbols
          || i `M.member` thSyms
          || i `M.member` fSyms
          || isAnyValue i -> do
          fail "variable"
    _ -> var <$> freshVar i

-- when -- NOTE: do not allow variables named as values
--   (isAnyValue i)
--   ( fail $ "variable " +| i |+ " cannot be named as a value from some other theory."
--   )
-- when -- NOTE: do not allow variables starting with a digit
--   (isDigit $ DT.head $ fromIdentifier i)
--   ( fail $ "variable " +| i |+ " is not allowed to start with a digit."
--   )
-- when -- NOTE: do not allow variables containing either "#x" or "#b"
--   (("#x" `DT.isInfixOf` fromIdentifier i) || ("#b" `DT.isInfixOf` fromIdentifier i))
--   ( fail $ "variable " +| i |+ " is not allowed to contain \"#x\" or \"#b\"."
--   )

getSortAnn :: Identifier -> Int -> Parser SortAnnotation
getSortAnn ident arity = do
  inp <- get
  case valueSort inp ident of
    Just sort | valueCheck inp (ident, sortAnnotation [] sort) -> return $ sortAnnotation [] sort
    _ -> do
      fS <- gets fSorts
      tS <- gets thSorts
      case M.lookup ident fS of
        Just sa -> return sa
        Nothing -> case M.lookup ident tS of
          Nothing -> fmtError $ "sort annotation for symbol " +| ident |+ " missing."
          Just [] -> fmtError $ "sort annotation for symbol " +| ident |+ " missing."
          -- NOTE: we have to pad the sort annotation because of chaining etc.
          Just [(sa, _, _)] -> expandSortAnnotation sa arity
          Just (_ : _) -> freshPSort arity
 where
  allEqual [] = True
  allEqual [_] = True
  allEqual (el : elems) = go el elems
   where
    go _ [] = True
    go e (e' : es) | e == e' = go e' es
    go _ _ = False

  expandSortAnnotation sa arity | length (inSort sa) == arity = return sa
  expandSortAnnotation sa arity
    | length (inSort sa) < arity && allEqual (inSort sa) =
        case inSort sa of
          [] -> fmtError "constant symbol applied to wrong number of arguments."
          inS@(so : _) -> do
            let expandedInSort =
                  inS ++ replicate (arity - length inS) so
            return $ sortAnnotation expandedInSort (outSort sa)
  expandSortAnnotation _ _ =
    fmtError $
      "parsed "
        +| ident
        |+ " cannot be attached with a meaningful sort. \
           \ Maybe {left,right}-associativity, chaining or pairwise is used."

freshPSort :: Int -> Parser SortAnnotation
freshPSort a = do
  sortAnnotation -- combine into sort annotation
    <$> (map makePSort <$> replicateM a lfreshI) -- fresh input sorts
    <*> (makePSort <$> lfreshI) -- fresh output sort
 where
  lfreshI = lift $ lift freshI
  makePSort = psort . ('?' `DT.cons`) . pack . show

-- parsers for sorts
pSort :: (Sort -> Bool) -> Parser Sort
pSort isKnownSort = do
  s <- try pTheorySort <|> litSort <$> sortIdent
  if isKnownSort s
    then return s
    else fmtError $ "sort " +| s |+ " unknown."
 where
  pTheorySort =
    (litSort <$> symbol "Int")
      <|> (litSort <$> symbol "Bool")
      <|> (litSort <$> symbol "Real")
      <|> pBitVec

  pBitVec = parens $ do
    _ <- symbol "_"
    n <- symbol "BitVec"
    attrsort n . AttrInt <$> numeral

pVarSortDeclarations :: (Sort -> Bool) -> Parser [(UnsortedVId, Sort)]
pVarSortDeclarations isKnownSort = many $ parens pVarSortDeclaration
 where
  pVarSortDeclaration = (,) <$> (freshVar =<< ariIdentifier) <*> pSort isKnownSort

----------------------------------------------------------------------------------------------------
-- not used anymore, but maybe useful in the future
----------------------------------------------------------------------------------------------------

-- we will not implement a parser covering all the allowed SMTLIB
-- formulas in the define-fun command
-- we will just implement a balanced parenthesis parser and then return
-- the whole formula as Text
pBalancedSMTTerm :: Parser Text
pBalancedSMTTerm =
  lexeme (try noParens <|> parensStable go)
 where
  go :: Parser Text
  go = fold <$> many (try noParens <|> parensStable go)

  noParens :: Parser Text
  noParens = pack <$> ((:) <$> noParen <*> many noParen)

  noParen :: Parser Char
  noParen = noneOf ("()" :: String)

  parensStable :: Parser Text -> Parser Text
  parensStable p = do
    op <- char '('
    v <- p
    cp <- char ')'
    return $ pack [op] <> v <> pack [cp]

----------------------------------------------------------------------------------------------------
-- formatter for parse errors
----------------------------------------------------------------------------------------------------

fmtParseError :: Builder -> String
fmtParseError = fmt . nameF "Parser: "

fmtError :: Builder -> a
fmtError = error . fmtParseError
