{- |
Module      : Box
Description :
Copyright   : (c) Jonas Schöpf, 2024
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides a wrapper to pretty print
with a specific given width and height.
-}
module Pretty.Box (Box, width, height, content, box, boxString, boxShow, boxPretty, onTopOf, belowOf, leftOf, rightOf) where

import Data.List (intercalate)
import Prettyprinter (Pretty (pretty))

-- | type for printing in box with specific measures
data Box = Box
  { width :: Int
  -- ^ width of the box
  , height :: Int
  -- ^ height of the box
  , content :: [String]
  -- ^ content of the box
  }

instance Show Box where
  show box =
    let
      h = height box
      w = width box
      c = content box
      vPadding = replicate (h - length c) (replicate w ' ')
      hPadding s = replicate (w - length s) ' ' ++ s
    in
      intercalate "\n" (vPadding ++ map hPadding c)

instance Pretty Box where
  pretty = pretty . show

-- | 'box' @width@ @height@ @content@ creates a 'Box'
box :: Int -> Int -> [String] -> Box
box = Box

-- | 'box' @width@ @height@ @content@ creates a 'Box'
boxString :: String -> Box
boxString s = box (length s) 1 [s]

-- | 'boxShow' @width@ @height@ @content@ executes 'show' on @content@ and creates a 'Box'
boxShow :: (Show a) => Int -> Int -> [a] -> Box
boxShow w h = box w h . map show

-- | 'boxPretty' @width@ @height@ @content@ executes 'pretty' on @content@ and creates a 'Box'
boxPretty :: (Pretty a, Show a) => Int -> Int -> [a] -> Box
boxPretty w h = box w h . map (show . pretty)

-- | 'onTopOf' @b1@ @b2@ stacks @b1@ on top of @b2@
onTopOf :: Box -> Box -> Box
onTopOf box1 box2 =
  box
    (max (width box1) (width box2))
    (height box1 + height box2)
    (content box1 ++ content box2)

-- | 'belowOf' @b1@ @b2@ puts @b1@ below of @b2@
belowOf :: Box -> Box -> Box
belowOf = flip onTopOf

-- | 'leftOf' @b1@ @b2@ puts @b1@ to the left of @b2@
leftOf :: Box -> Box -> Box
leftOf box1 box2 =
  box
    (width box1 + width box2)
    (max (height box1) (height box2))
    (go (content box1) (content box2))
 where
  go cs1 cs2
    | length cs1 > length cs2 =
        let (header, rem) = splitAt (length cs1 - length cs2) cs1
        in  map (++ replicate (width box2) ' ') header ++ go rem cs2
    | length cs1 < length cs2 =
        let (header, rem) = splitAt (length cs2 - length cs1) cs2
        in  map (replicate (width box1) ' ' ++) header ++ go cs1 rem
    | otherwise = zipWith (++) cs1 cs2

-- | 'rightOf' @b1@ @b2@ puts @b1@ to the right of @b2@
rightOf :: Box -> Box -> Box
rightOf = flip leftOf
