{- |
Module      : ConstrainedNormalForm
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides functions to perform rewriting on constrained terms.  This
covers, single steps, reflexive transitive steps, parallel steps and multisteps.
-}
module Rewriting.ConstrainedRewriting.ConstrainedNormalForm where

import Data.LCTRS.FIdentifier (FId)
import Data.LCTRS.Guard (
  createGuard,
 )
import Data.LCTRS.LCTRS (LCTRS, getRules)
import Data.LCTRS.Rule (
  CTerm,
 )
import Data.LCTRS.Sort (Sort, Sorted)
import qualified Data.LCTRS.Term as T
import Data.Maybe (isNothing)
import Data.Monad
import Data.SExpr
import Prettyprinter (Pretty (..))
import Rewriting.ConstrainedRewriting.ConstrainedRewriting (
  isReducibleInstance,
  matchesAnyPos,
 )
import Rewriting.ConstrainedRewriting.SingleStepRewriting (isCalculationPossible)
import Rewriting.SMT (
  satGuard,
 )
import Utils (findM)

----------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------
-- normal form checking on constrained terms
----------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------

isNormalFormIn
  :: (Ord v, Pretty v, Pretty f, Ord f, ToSExpr f, ToSExpr v, Sorted v)
  => (Sort -> Int -> v)
  -> LCTRS (FId f) v
  -> T.Term (FId f) v
  -> StateM (Maybe Bool)
isNormalFormIn freshVar lc term = isCNormalFormIn freshVar lc (term, createGuard [])

isCNormalFormIn
  :: (Ord v, Pretty v, Pretty f, Ord f, ToSExpr f, ToSExpr v, Sorted v)
  => (Sort -> Int -> v)
  -> LCTRS (FId f) v
  -> CTerm (FId f) v
  -> StateM (Maybe Bool)
isCNormalFormIn freshVar lc (term, c) = do
  satisfiable <- satGuard c
  case satisfiable of
    Just True -> do
      calcPossible <- isCalculationPossible freshVar lc (term, c)
      if calcPossible
        then return (Just False)
        else do
          matchings <- mapM (matchesAnyPos subterms c) $ getRules lc
          case mconcat matchings of
            [] -> return (Just True)
            instances -> do
              Just . isNothing <$> findM (isReducibleInstance c) instances
    _ -> return Nothing
 where
  subterms = T.subterms term
