{- |
Module      : Renaming
Description :
Copyright   : (c) Jonas Schöpf, 2024
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable


This module provides types and functions for Renamings.
-}
module Rewriting.Renaming where

import Control.Monad.State.Strict (evalStateT, gets, modify)
import Data.LCTRS.Sort (Sorted (sort))
import Data.LCTRS.Term (Term (..), fun, var)
import Data.LCTRS.VIdentifier (VId, freshV)
import qualified Data.Map.Strict as M
import Data.Monad (MonadFresh (freshInt))
import Data.SExpr (ToSExpr (toSExpr))
import Prettyprinter (Pretty (pretty))
import qualified SimpleSMT as SMT

----------------------------------------------------------------------------------------------------
-- special type for renamings
----------------------------------------------------------------------------------------------------

newtype Renaming v v' = Renaming {unpackRenaming :: Either v v'}
  deriving (Eq, Ord)

instance (Pretty v, Pretty v') => Pretty (Renaming v v') where
  pretty (Renaming (Left v)) = pretty v <> "'" -- <> "_" <> "l"
  pretty (Renaming (Right v')) = pretty v' <> "\"" -- <> "r"

instance (Pretty v, Pretty v') => ToSExpr (Renaming v v') where
  toSExpr (Renaming (Left v)) = SMT.const $ show (pretty v) ++ "_" ++ "L"
  toSExpr (Renaming (Right v)) = SMT.const $ show (pretty v) ++ "_" ++ "R"

instance (Sorted v, Sorted v') => Sorted (Renaming v v') where
  sort (Renaming (Left v)) = sort v
  sort (Renaming (Right v)) = sort v

eitherToRenaming :: Either v v' -> Renaming v v'
eitherToRenaming = Renaming

innerRenaming :: v -> Renaming v v'
innerRenaming = Renaming . Left
outerRenaming :: v -> Renaming v' v
outerRenaming = Renaming . Right

removeRenamingSafely
  :: (MonadFresh m, Ord v1, Ord v2) => Term f (Renaming (VId v1) (VId v2)) -> m (Term f (VId v3))
removeRenamingSafely term = evalStateT (go term) M.empty
 where
  go (Var v) = var <$> freshVar v
  go (Fun typ f args) = fun typ f <$> mapM go args

  freshVar v = do
    cached <- gets (M.lookup v)
    case cached of
      Just var -> return var
      Nothing -> do
        val <- freshV (sort v) <$> freshInt
        modify $ M.insert v val
        return val
