{-# LANGUAGE BangPatterns #-}

{- |
{\-# LANGUAGE BangPatterns #-\}

Module      : Rewrite
Description :
Copyright   : (c) Jonas Schöpf, 2023
License     : GPL-3
Maintainer  : jonas.schoepf@uibk.ac.at
Stability   : stable

For now, this module is from the term-rewriting library, but allows rules with
variables occuring on the right-hand side. This will be replaced by another
module in the future.
-}
module Rewriting.Rewrite (
  Reduct (..),
  fullRewrite,
  outerRewrite,
  innerRewrite,
  rootRewrite,
) where

import Data.LCTRS.Position (Pos, epsilon, prepend)
import Data.LCTRS.Rule (Rule (lhs, rhs))
import Data.LCTRS.Term (Term (..), fun)
import Data.Maybe (maybeToList)
import Rewriting.Substitution (Subst, apply, match)

{- | A reduct. It contains the resulting term, the position that the term
was rewritten at, and the applied rule.
-}
data Reduct f v = Reduct
  { result :: Term f v
  , pos :: Pos
  , rule :: Rule f v
  , subst :: Subst f v
  }

-- | A rewrite strategy.
type Strategy f v = Term f v -> [Reduct f v]

{- | Full rewriting: Apply rules anywhere in the term.

Reducts are returned in pre-order: the first is a leftmost, outermost redex.
-}
fullRewrite
  :: (Ord v, Eq v, Eq f)
  => [Rule f v]
  -> Strategy f v
fullRewrite trs t = rootRewrite trs t ++ nested (fullRewrite trs) t

{- | Outer rewriting: Apply rules at outermost redexes.

Reducts are returned in left to right order.
-}
outerRewrite
  :: (Ord v, Eq v, Eq f)
  => [Rule f v]
  -> Strategy f v
outerRewrite trs t = case rootRewrite trs t of
  [] -> nested (outerRewrite trs) t
  rs -> rs

{- | Inner rewriting: Apply rules at innermost redexes.

Reducts are returned in left to right order.
-}
innerRewrite
  :: (Ord v, Eq v, Eq f)
  => [Rule f v]
  -> Strategy f v
innerRewrite trs t = case nested (innerRewrite trs) t of
  [] -> rootRewrite trs t
  rs -> rs

{- | Root rewriting: Apply rules only at the root of the term.

This is mainly useful as a building block for various rewriting strategies.
-}
rootRewrite
  :: (Ord v, Eq v, Eq f)
  => [Rule f v]
  -> Strategy f v
rootRewrite trs t = do
  r <- trs
  s <- maybeToList $ match (lhs r) t
  let t' = apply s (rhs r)
  return Reduct{result = t', pos = epsilon, rule = r, subst = s}

-- gApply :: (Ord v) => Subst f v -> Term f v -> Term f v
-- gApply subst = foldTerm var fun
--  where
--   var v = M.findWithDefault (var v) v (toMap subst)

{- | Nested rewriting: Apply a rewriting strategy to all arguments of a
function symbol, left to right. For variables, the result will be empty.

This is another building block for rewriting strategies.
-}
nested :: Strategy f v -> Strategy f v
nested _ (Var _) = []
nested s (Fun typ f ts) = do
  (n, cl, t) <- listContexts ts
  (\r -> r{result = fun typ f (cl (result r)), pos = n `prepend` pos r}) `fmap` s t

{- | Return a list of contexts of a list. Each returned element is an element
index (starting from 0), a function that replaces the list element by a
new one, and the original element.
-}
listContexts :: [a] -> [(Int, a -> [a], a)]
listContexts = go 0 id
 where
  go _ _ [] = []
  go !n f (x : xs) = (n, f . (: xs), x) : go (n + 1) (f . (x :)) xs
