module Demo08 where

import Text.ParserCombinators.Parsec

import System.Directory (doesDirectoryExist, getDirectoryContents)
import System.FilePath ((</>))
import Control.Monad (when)
import Control.Monad.Writer 
import Control.Monad.State
import Control.Monad.Reader
import Data.Bifunctor(second)


pQuadratic = try (string "ab" >> pQuadratic)
   <|> (eof >> return "(ab)^*")
   <|> (many (oneOf "ab") >> string "c" >> return "end in c")

pLinear = (try (string "ab") >> pLinear)
   <|> (eof >> return "(ab)^*")
   <|> (many (oneOf "ab") >> string "c" >> return "end in c")

testQ n = parse pQuadratic "" $ concat (replicate n "ab") ++ "a"
testL n = parse pLinear    "" $ concat (replicate n "ab") ++ "a"



listDirectory :: FilePath -> IO [String]
listDirectory d = filter notDots <$> getDirectoryContents d
    where notDots p = not $ p `elem` [".", ".."]

-- just using IO monad
countEntries1 :: FilePath -> IO [(FilePath, Int)]
countEntries1 path = do
  contents <- listDirectory path
  rest <- flip mapM contents $ \name -> do
      let newName = path </> name
      isDir <- doesDirectoryExist newName
      if isDir
        then countEntries1 newName
        else return []
  return $ (path, length contents) : concat rest
  
-- let us use writer monad as well via writer monad transformer
countEntries2Main :: FilePath -> WriterT [(FilePath, Int)] IO ()
countEntries2Main path = do
  contents <- liftIO . listDirectory $ path
  tell [(path, length contents)]
  flip mapM_ contents $ \name -> do
      let newName = path </> name
      isDir <- liftIO . doesDirectoryExist $ newName
      when isDir $ countEntries2Main newName

countEntries2 :: FilePath -> IO [(FilePath, Int)]      
countEntries2 = fmap snd . runWriterT . countEntries2Main



-- stack of monad transformers

data AppConfig = AppConfig {
      cfgMaxDepth :: Int
    } deriving (Show)

data AppState = AppState {
      stDeepestReached :: Int
    } deriving (Show)
    
type App = ReaderT AppConfig (StateT AppState IO)

runApp :: App a -> Int -> IO (a, AppState)
runApp app maxDepth =
    let config = AppConfig maxDepth
        state = AppState 0
    in runStateT (runReaderT app config) state
    
countEntries3Main :: Int -> FilePath -> App [(FilePath, Int)]
countEntries3Main curDepth path = do
  contents <- liftIO . listDirectory $ path
  allowedDepth <- cfgMaxDepth <$> ask
  rest <- flip mapM contents $ \name -> do
    let newPath = path </> name
    isDir <- liftIO $ doesDirectoryExist newPath
    if isDir && curDepth < allowedDepth
      then do
        let newDepth = curDepth + 1
        st <- get
        when (stDeepestReached st < newDepth) $
          put $ st { stDeepestReached = newDepth }
        countEntries3Main newDepth newPath
      else return []
  return $ (path, length contents) : concat rest
  
countEntries3 :: Int -> FilePath -> IO ([(FilePath, Int)], Int)
countEntries3 md fp = second stDeepestReached <$> runApp (countEntries3Main 0 fp) md


--- lifting

class Monad m => MyMonad m where
  myFun :: Int -> a -> m [a]
  
foo :: MyMonad m => a -> ReaderT Int m a
foo x = do 
  i <- ask 
  xs <- lift $ myFun i x
  return $ xs !! max i 5

bar :: StateT Int (StateT String IO) ()
bar = do 
  (x :: Int) <- read <$> liftIO getLine
  put x                         -- outer StateT
  (s :: String) <- lift $ get   -- inner StateT
  liftIO $ putStrLn s
