module Main where

import Data.List
import Data.Time.Clock (diffUTCTime, getCurrentTime)
import System.Environment (getArgs)
import System.Random (StdGen, getStdGen, randoms)
import System.IO
import Control.DeepSeq(force)
import Control.Exception(evaluate)
import Text.Printf
import GHC.Conc (numCapabilities)
import Control.Parallel.Strategies
import Control.Monad(replicateM_)
import Control.Concurrent

import Demo10_Logger


fib :: Int -> Integer
fib 0 = 1
fib 1 = 1
fib n = fib (n - 1) + fib (n - 2)

printTimeSince t0 = do
  t1 <- getCurrentTime
  printf "time: %.2fs\n" (realToFrac (diffUTCTime t1 t0) :: Double)


mainFib n = do
  let test = [test1,test2,test3] !! (read n - 1)
  t0 <- getCurrentTime
  r <- evaluate (runEval test)
  printTimeSince t0
  print r
  printTimeSince t0

test1 = do
  x <- rpar (fib 37)
  y <- rpar (fib 36)
  return (x,y)

test2 = do
  x <- rpar (fib 37)
  y <- rseq (fib 36)
  return (x,y)

test3 = do
  x <- rpar (fib 37)
  y <- rpar (fib 36)
  rseq x
  rseq y
  return (x,y)


qsortSeq :: Ord a => [a] -> [a]
qsortSeq (x : xs) = let
  (low, high) = partition (< x) xs
  sLow = qsortSeq low
  sHigh = qsortSeq high
 in sLow ++ [x] ++ sHigh
qsortSeq [] = []

qsortPar1 :: Ord a => [a] -> [a]
qsortPar1 (x : xs) = let 
  (low, high) = partition (< x) xs
 in runEval $ do
       sLow <- rpar $ qsortPar1 low
       sHigh <- rpar $ qsortPar1 high
       rseq $ sLow
       rseq $ sHigh
       return $ sLow ++ [x] ++ sHigh
qsortPar1 [] = []

spine (_ : xs) = spine xs
spine [] = ()

qsortPar2 :: Ord a => [a] -> [a]
qsortPar2 (x : xs) = let 
  (low, high) = partition (< x) xs
 in runEval $ do
       sLow <- rpar $ qsortPar2 low
       sHigh <- rpar $ qsortPar2 high
       rseq $ spine sLow
       rseq $ spine sHigh
       return $ sLow ++ [x] ++ sHigh
qsortPar2 [] = []


qsortPar3 :: Ord a => [a] -> [a]
qsortPar3 = qsortPar3Main (10 :: Int)
qsortPar3Main d xs 
  | d == 0 = qsortSeq xs
qsortPar3Main d (x : xs) = let 
  (low, high) = partition (< x) xs
 in runEval $ do
       sLow <- rpar $ qsortPar3Main (d-1) low
       sHigh <- rpar $ qsortPar3Main (d-1) high
       rseq $ spine sLow
       rseq $ spine sHigh
       return $ sLow ++ [x] ++ sHigh
qsortPar3Main _ [] = []
 
randomInts :: Int -> StdGen -> [Int]
randomInts k g = take k (randoms g)
                 
sortAlgs :: [(String, [Int] -> [Int])]
sortAlgs = [
  ("qsortSeq",qsortSeq)
  , ("qsortPar1",qsortPar1)
  , ("qsortPar2",qsortPar2)
  , ("qsortPar3",qsortPar3)
  ]
  
sortFile = "numbers.txt"

generateNumbers :: String -> IO ()
generateNumbers num = do 
  let n = (read num :: Int)
  numbers <- randomInts n `fmap` getStdGen
  writeFile sortFile $ unlines $ map show numbers
  
mainSort :: String -> IO ()                 
mainSort algName = do
  case lookup algName sortAlgs of
    Nothing -> error $ "unknown sorting algorithm, choose " ++ show (map fst sortAlgs)
    Just sortAlg -> do
      input <- lines <$> readFile sortFile 
--      let numbers = map read input 
--      let numbers = force $ map read input 
      let numbers = force $ (map read input `using` parList rseq)
      putStrLn $ "We have " ++ show (length numbers) ++ " elements to sort."
      start <- getCurrentTime
      let sorted = sortAlg numbers
      putStrLn $ "Sorted all " ++ show (length sorted) ++ " elements."
      end <- getCurrentTime
      putStrLn $ show (end `diffUTCTime` start) ++ " elapsed."

mainThreadDemo1 = do
  hSetBuffering stdout NoBuffering
  _ <- forkIO (replicateM_ 100000 (putChar 'a'))
  replicateM_ 100000 (putChar 'b')

mainThreadDemo2 = do
  s <- getLine
  if s == "exit" 
    then return ()
    else do 
      _ <- forkIO $ setReminder s
      mainThreadDemo2
  
setReminder :: String -> IO ()
setReminder s = do
  let t = read s :: Int
  putStrLn $ "Reminder in " ++ show t ++ " seconds" 
  threadDelay $ 10^(6 :: Int) * t
  putStrLn $ "Reminder of " ++ show t ++ " seconds is over! \BEL"
  
comm1 = do 
  m <- newEmptyMVar
  _ <- forkIO $ putMVar m 'x'
  r <- takeMVar m
  print r

comm2 = do 
  m <- newEmptyMVar
  _ <- forkIO $ do { putMVar m 'x'; putMVar m 'y' }
  r <- takeMVar m
  print r
  r <- takeMVar m
  print r

comm3 = do 
  m <- newEmptyMVar
  n <- newEmptyMVar
  _ <- forkIO $ do { s <- takeMVar m; putMVar n (s + 1) }
  r <- takeMVar n
  putMVar m (42 :: Int)
  print r

message s i = "message " ++ show i ++ " of " ++ s

mainLogger = do 
  l <- initLogger 
  _ <- forkIO $ mapM_ (logMessage l . message "fork 1") [1..100]
  _ <- forkIO $ mapM_ (logMessage l . message "fork 2") [1..100]
  mapM_ (logMessage l . message "main thread") [1..100]
  logStop l


main = do
  putStrLn $ "number of cores: " ++ show numCapabilities
  args <- getArgs
  putStrLn $ "args: " ++ show args
  case args of 
    ["fib", alg]  -> mainFib alg
    ["sort", alg]  -> mainSort alg
    ["numbers", n] -> generateNumbers n
    ["td1"] -> mainThreadDemo1
    ["td2"] -> mainThreadDemo2
    ["comm1"] -> comm1
    ["comm2"] -> comm2
    ["comm3"] -> comm3
    ["logger"] -> mainLogger
    _ -> error "unknown invocation"
