module Main where

import Data.List(partition)
import Data.Time.Clock (diffUTCTime, getCurrentTime)
import System.Environment (getArgs)
import System.Random (getStdGen, randoms)
import System.IO
import Control.DeepSeq(force)
import Control.Parallel.Strategies
import Control.Monad
import Control.Concurrent
import Control.Concurrent.Async
import qualified Data.Map.Strict as M
import qualified Data.ByteString.Char8 as BSC
import Prelude hiding (lookup)
import qualified Prelude



{- main function selects between sort, numbers and cmap functionality -}  

main = do
  numCores <- getNumCapabilities
  putStrLn $ "number of cores: " ++ show numCores
  args <- getArgs
  putStrLn $ "args: " ++ show args
  case args of 
    ["sort", inputAlg, sortAlg] -> mainSort inputAlg sortAlg
    ["numbers", n] -> generateNumbers n
    ["cmap", n, m, t] -> mainCMapTest n m t
    _ -> error "unknown invocation, choose one of sort iAlg sAlg|numbers n|cmap numElemsPerThread costElem numThreads"


{- Algorithms from the lecture -}

sortFile = "numbers.txt"
 
generateNumbers :: String -> IO ()
generateNumbers num = do 
  let n = (read num :: Int)
  numbers <- randomInts n <$> getStdGen
  writeFile sortFile $ unlines $ map show numbers
  where randomInts n g = (take n (randoms g) :: [Int])

seqInput = do 
  input <- lines <$> readFile sortFile 
  let numbers = force $ map read input
  return numbers

parListInput = do 
  input <- lines <$> readFile sortFile 
  let numbers = force $ (map read input `using` parList rseq)
  return numbers

inputAlgs :: [(String, IO [Int])]
inputAlgs = [
  ("seqInput",seqInput)
  , ("parListInput",parListInput)
  , ("optimizedInput",optimizedInput)
  ]

qsortSeq :: Ord a => [a] -> [a]
qsortSeq (x : xs) = let
  (low, high) = partition (< x) xs
  sLow = qsortSeq low
  sHigh = qsortSeq high
 in sLow ++ [x] ++ sHigh
qsortSeq [] = []

spine (_ : xs) = spine xs
spine [] = ()

qsortPar :: Ord a => [a] -> [a]
qsortPar = qsortParMain (10 :: Int)
qsortParMain d xs 
  | d == 0 = qsortSeq xs
qsortParMain d (x : xs) = let 
  (low, high) = partition (< x) xs
 in runEval $ do
       sLow <- rpar $ qsortParMain (d-1) low
       sHigh <- rpar $ qsortParMain (d-1) high
       rseq $ spine sLow
       rseq $ spine sHigh
       return $ sLow ++ [x] ++ sHigh
qsortParMain _ [] = []
 
sortAlgs :: [(String, Int -> [Int] -> [Int])]
sortAlgs = [
  ("qsortSeq", const qsortSeq)
  , ("qsortPar", const qsortPar)
  , ("hybridSort", hybridSort)
  ]


{- wrapper to invoke the various input readers and sorting algorithms -}  

mainSort :: String -> String -> IO ()                 
mainSort inputAlgName sortAlgName = case Prelude.lookup inputAlgName inputAlgs of 
  Nothing -> error $ "unknown input reader algorithm, choose " ++ show (map fst inputAlgs)
  Just inputAlg -> case Prelude.lookup sortAlgName sortAlgs of
    Nothing -> error $ "unknown sorting algorithm, choose " ++ show (map fst sortAlgs)    
    Just sortAlg ->  do
      start <- getCurrentTime
      numbers <- inputAlg
      putStrLn $ "We have " ++ show (length numbers) ++ " elements to sort."
      mid <- getCurrentTime
      putStrLn $ show (mid `diffUTCTime` start) ++ " for reading input phase"
      numCores <- getNumCapabilities
      let sorted = sortAlg numCores numbers
      putStrLn $ "Sorted all " ++ show (length sorted) ++ " elements."
      end <- getCurrentTime
      putStrLn $ show (end `diffUTCTime` mid) ++ " for sorting phase"
      putStrLn $ show (end `diffUTCTime` start) ++ " total time"



{- Task 1  -}

hybridSort :: Ord a => Int -> [a] -> [a]
hybridSort nc xs = undefined


{- Task 2 -}

optimizedInput :: IO [Int]
optimizedInput = do 
  h <- openFile sortFile ReadMode
  (size :: Int) <- fromInteger <$> hFileSize h
  nc <- getNumCapabilities
  undefined



{- Task 3 -}

data CMap a b = Undefined

empty :: IO (CMap a b)
empty = undefined

-- note that the modified map is not returned
insert :: Ord a => CMap a b -> a -> b -> IO ()
insert = undefined

lookup :: Ord a => CMap a b -> a -> IO (Maybe b)
lookup = undefined


-- testing program
mainCMapTest n ms th = do 
  let numElems = (read n :: Int)
  let numThreads = (read th :: Int)
  let m = (read ms :: Int)
  cmap <- empty 
  -- each thread inserts numElems elements into the concurrent map,
  -- and the main thread waits on completion of all these insertions
  forConcurrently_ [1..numThreads] ( \ i -> do
       forM_ [ i .. i + numElems] 
          (\ x -> let fx = sum [i+x .. i+x+m] in fx `seq` insert cmap x fx)          
    )
  -- afterwards, the main thread reads and prints some results
  firstEntries <- mapM (lookup cmap) [0..10]
  putStrLn $ show firstEntries
    
