module Main(main) where

import System.Environment
import Text.Printf

main = do
    [version, d] <- getArgs
    let mean = case version of 
         "1" -> mean1
         "2" -> mean2
         _ -> error "unknown version, choose 1 or 2" 
    printf "%f\n" (mean [1 .. read d])

mean1 :: [Double] -> Double
mean1 xs = ({-# SCC "sum" #-} sum xs) / fromIntegral ({-# SCC "len" #-} length xs)

mean2 :: [Double] -> Double
mean2 xs = let (s, l) = foldl' step (0, 0) xs in s / fromIntegral l
  where 
    step :: (Double, Int) -> Double -> (Double, Int)
    step (s, l) x = let 
        s' = s + x
        l' = l + 1
      in s' `seq` l' `seq` (s', l')

-- simple queues
-- Queue1 b e represents queue with elements b ++ reverse e
-- (b = beginning part, e = ending part)
data Queue1 a = Queue1 [a] [a]

empty1 :: Queue1 a
empty1 = Queue1 [] []

insert1 :: a -> Queue1 a -> Queue1 a
insert1 x (Queue1 b e) = Queue1 b (x : e)

remove1 :: Queue1 a -> (a, Queue1 a)
remove1 (Queue1 (x : b) e) = (x, Queue1 b e)
remove1 (Queue1 [] []) = error "empty queue"
remove1 (Queue1 [] e) = remove1 (Queue1 (reverse e) [])


-- Okasaki's queues as paired lazy lists

-- the integers are the lengths of b and e
-- invariant: le <= lb
data Queue2 a = Queue2 Int [a] Int [a]

empty2 :: Queue2 a
empty2 = Queue2 0 [] 0 []

insert2 :: a -> Queue2 a -> Queue2 a
insert2 x (Queue2 lb b le e) = let 
   lxe = le + 1 
 in makeQ2 lb b lxe (x : e)

remove2 :: Queue2 a -> (a, Queue2 a)
remove2 (Queue2 _ [] _ _) = error "empty queue"
remove2 (Queue2 lxb (x : b) le e) = let 
   lb = lxb - 1
   newQ = makeQ2 lb b le e 
 in seq newQ (x, newQ)

-- assumes le <= lb + 1
makeQ2 :: Int -> [a] -> Int -> [a] -> Queue2 a 
makeQ2 lb b le e 
  | le <= lb = Queue2 lb b le e
  | otherwise = let lbe = lb + le      -- le = lb + 1
      in Queue2 lbe (rot b e []) 0 []

-- rot b e a = b ++ reverse e ++ a, assumes length e = length b + 1
rot :: [a] -> [a] -> [a] -> [a]
rot [] [x] a = x : a
rot (x : b) (y : e) a = x : rot b e (y : a)


-- Okasaki's queues with pre-evaluation
-- (O(1) worst case)
-- Queue3 b e b' where b' is a sublist of b and will be used to
--   force evaluation; 
--   invariants: le <= lb, lb' = lb - le
data Queue3 a = Queue3 [a] [a] [a]

empty3 :: Queue3 a
empty3 = Queue3 [] [] []

insert3 :: a -> Queue3 a -> Queue3 a
insert3 x (Queue3 b e b') = makeQ3 b (x : e) b'

remove3 :: Queue3 a -> (a, Queue3 a)
remove3 (Queue3 [] _ _) = error "empty queue"
remove3 (Queue3 (x : b) e b') = let
   newQ = makeQ3 b e b' 
 in seq newQ (x, newQ)

makeQ3 :: [a] -> [a] -> [a] -> Queue3 a 
makeQ3 b e (_ : b') = Queue3 b e b'
makeQ3 b e [] = let b' = rot b e [] in Queue3 b' [] b'
 
