{-# LANGUAGE GADTs, GeneralizedNewtypeDeriving #-}

module Tests_04 (runTests, boolTests) where

import Data.List (isInfixOf)
import Template_04
import Test.LeanCheck

-- Generic Setup
data Test = forall a. (Testable a) => Test String String a

runTests =
  flip
    mapM_
    tests
    (\(Test ex name t) -> putStrLn ("running " ++ ex ++ "(" ++ name ++ ")" ++ "-tests") >> checkFor 1000 t)

------------------------------------------------------------
-- Tests für Template_04
------------------------------------------------------------

-- 1. divSafeList: Länge der Ergebnisliste stimmt
divSafeListLength :: [(Double, Double)] -> Bool
divSafeListLength xs = length (divSafeList xs) == length xs

-- 2. removeMaybe: entfernt Nothing korrekt
removeMaybeRemoves :: [Maybe Double] -> Bool
removeMaybeRemoves xs =
  all (/= Nothing) (map Just (removeMaybe xs))

removeMaybeKeepsJust :: [Double] -> Bool
removeMaybeKeepsJust xs =
  removeMaybe (map Just xs) == xs

-- 3. find: Key gefunden / nicht gefunden
findKeyFound :: (Eq a, Eq b) => a -> b -> [(a, b)] -> Bool
findKeyFound k v xs = find k ((k,v):xs) == Right v

findKeyNotFound :: (Eq a, Eq b) => a -> [(a,b)] -> Bool
findKeyNotFound k xs = find k xs == Left "Key was not found"

-- 4. triples: erzeugt überlappende Tripel
triplesTest :: [Int] -> Bool
triplesTest xs =
  let ts = triples xs
  in all (\(a,b,c) -> [a,b,c] `isInfixOf` xs) ts
     && length ts == max 0 (length xs - 2)

-- Tests for expressions

data Vars = X | Y | Z deriving (Eq)

instance Show Vars where
  show X = "x"
  show Y = "y"
  show Z = "z"
  
newtype SmallNums = SmallNum Integer deriving (Eq,Num)

instance Listable SmallNums where
  tiers = [map SmallNum [5,9,17,-3,0,1]]
  
instance Show SmallNums where
  show (SmallNum n) = show n

instance Listable Vars where
  tiers = [[X,Y,Z]]


instance (Listable a, Listable b) => Listable (Expr a b) where
  tiers = cons1 Var \/ cons1 Number \/ cons2 Plus
  
foldE n v p (Number z) = n z
foldE n v p (Var x) = v x
foldE n v p (Plus e1 e2) = p (foldE n v p e1) (foldE n v p e2)
  
-- 5. eval is evaluation
evalTest :: Expr Vars SmallNums -> [(Vars,SmallNums)] -> Bool
evalTest e a = case foldE Just (flip lookup a) 
   (\ e1 e2 -> do { x1 <- e1; x2 <- e2; return $ x1 + x2}) e of
  Nothing -> True
  Just r -> r == eval e a
  
-- 6. unknownVar tests
unknownVarTest :: Expr Vars SmallNums -> [(Vars,SmallNums)] -> Bool
unknownVarTest e a = 
  let 
    xe = foldE (const []) (\ x -> [x]) (++) e
    xa = map fst a
  in case unknownVar e a of
        Nothing -> all (\ x -> x `elem` xa) xe
        Just x -> x `elem` xe && not (x `elem` xa)
------------------------------------------------------------
-- Testsammlung
------------------------------------------------------------

tests :: [Test]
tests =
  [ Test "4.1" "divSafeList length" divSafeListLength
    , Test "4.2" "removeMaybe removes Nothing" removeMaybeRemoves
    , Test "4.2" "removeMaybe keeps Just" removeMaybeKeepsJust
    , Test "4.3" "find key found" (findKeyFound 'a' "foo" [])
    , Test "4.3" "find key not found" (findKeyNotFound 'x' ([] :: [(Char,String)]))
    , Test "4.4" "triples overlapping" triplesTest
    , Test "4.5" "eval is computing correct result" evalTest
    , Test "4.6" "unknownVar is computing correct result" unknownVarTest
  ]

boolTests :: [((String, String), Bool)]
boolTests = map (\(Test ex descr t) -> ((ex, descr), holds 1000 t)) tests