{-# LANGUAGE GADTs #-}
{-# LANGUAGE PartialTypeSignatures #-}

module Tests_06 (runTests, boolTests) where

import Template_06 (Polynom(..), normalize, MonoidC (..), VoteSeq (..), combine, normaliseVote)
import Test.LeanCheck

-- Generic Setup
data Test = forall a. (Testable a) => Test String String a

runTests =
  flip
    mapM_
    tests
    (\(Test ex name t) -> putStrLn ("running " ++ ex ++ "(" ++ name ++ ")" ++ "-tests") >> checkFor 1000 t)

-- Tests for this week

-- 1. Listable instance
instance Listable Polynom where
  tiers = cons1 Polynom  -- generiert Polynome aus Listen von Int

-- 2. Hilfsfunktionen
eqPoly (Polynom xs) (Polynom ys) = normalize (Polynom xs) == normalize (Polynom ys)
samePolyValue (Polynom xs) (Polynom ys) = eqPoly (normalize (Polynom xs)) (normalize (Polynom ys))

negatePoly s (Polynom xs) = Polynom (map (s *) xs)

validPoly (Polynom xs) = True  -- alle Int-Listen sind valide

-- 3. Normalize Tests
normalizeIdempotentTest p = validPoly p ==> eqPoly (normalize (normalize p)) (normalize p)
normalizeValueTest p = validPoly p ==> samePolyValue (normalize p) p

-- 4. Eq / Ord Tests
eqPolyTest p1 p2 = validPoly p1 ==> validPoly p2 ==> eqPoly p1 p2 == (normalize p1 == normalize p2)
eqPolyTest2 p1 = validPoly p1 ==> normalize p1 == p1
ordPolyTest p1 p2 = validPoly p1 ==> validPoly p2 ==> compare (normalize p1) (normalize p2) == compare p1 p2

-- 5. Num Tests
polyAddComm p1 p2 = validPoly p1 ==> validPoly p2 ==> p1 + p2 == p2 + p1
polyAddAssoc p1 p2 p3 = validPoly p1 ==> validPoly p2 ==> validPoly p3 ==> p1 + (p2 + p3) == (p1 + p2) + p3
polyMultComm p1 p2 = validPoly p1 ==> validPoly p2 ==> p1 * p2 == p2 * p1
polyMultAssoc p1 p2 p3 = validPoly p1 ==> validPoly p2 ==> validPoly p3 ==> p1 * (p2 * p3) == (p1 * p2) * p3
polyAddZero p = validPoly p ==> p + 0 == p
polyMultZero p = validPoly p ==> p * 0 == 0 && 0 * p == 0
polyMultOne p = validPoly p ==> p * 1 == p && 1 * p == p
polyAddNegate p = validPoly p ==> p + negate p == 0
polyDistrib p1 p2 p3 = validPoly p1 ==> validPoly p2 ==> validPoly p3 ==> p1 * (p2 + p3) == p1 * p2 + p1 * p3

-- 6. Show Tests
showPolyTest1 p = validPoly p ==> show (normalize p) == show (normalize p)
showPolyTest2 p1 p2 = validPoly p1 ==> validPoly p2 ==> eqPoly p1 p2 ==> show p1 == show p2



-------------------------------

newtype VoteBSeq = VSB [Bool]

toVoteSeq (VSB s) = VS (map (\b -> if b then 'U' else 'D') s)

instance Listable VoteBSeq where
  tiers = cons1 VSB

instance Show VoteBSeq where
  show s = case toVoteSeq s of VS l -> "(VS " ++ show l ++ ")"

-- 2.1
monoidCBinopAssoc n m o = binop n (binop m o) == binop (binop n m) o

monoidCNeutral n = binop n neutral == n && binop neutral n == n

integerMonoidLaws :: Integer -> Integer -> Integer -> Bool
integerMonoidLaws n m o = monoidCBinopAssoc n m o && monoidCNeutral n

boolMonoidLaws :: Bool -> Bool -> Bool -> Bool
boolMonoidLaws n m o = monoidCBinopAssoc n m o && monoidCNeutral n

listMonoidLaws :: [Int] -> [Int] -> [Int] -> Bool
listMonoidLaws n m o = monoidCBinopAssoc n m o && monoidCNeutral n

-- 2.2
numToString n
  | n < 0 = replicate (abs n) 'D'
  | otherwise = replicate n 'U'

numToVS :: Int -> VoteSeq
numToVS n = VS (numToString n)

blowUpString xs = concat $ replicate 3 "UD" ++ xs : replicate 3 "DD" ++ replicate 3 "UU"

normaliseTest n = let xs = numToString n in xs == normaliseVote (blowUpString xs)

isNormalised [] = True
isNormalised [x] = True
isNormalised (x : xs) = all (== x) xs

-- 2.3
voteSeqEqTest n = let xs = numToString n in VS xs == VS (blowUpString xs)

voteSeqDiffTest n m =
  let x = numToVS n
      y = numToVS m
   in n /= m ==> x /= y

voteSeqAdditionNormalized :: VoteBSeq -> VoteBSeq -> Bool
voteSeqAdditionNormalized x y = case binop (toVoteSeq x) (toVoteSeq y) of VS zs -> isNormalised zs

voteSeqAdditionCorrectResult n m =
  let x = numToVS n
      y = numToVS m
      z = numToVS (n + m)
   in binop x y == z

voteSeqNeutral n = let x = numToVS n in binop x neutral == x && binop neutral x == x

voteSeqShow n =
  let s = blowUpString (numToString n)
   in (show (numToVS n) == show (numToString n)) && (show (VS s) == show s)

-- 2.4
combineInteger :: [Integer] -> Bool
combineInteger xs = combine xs == foldl1 (*) (1 : xs)

combineBool :: [Bool] -> Bool
combineBool xs = combine xs == foldl1 (&&) (True : xs)

combineList :: [[Int]] -> Bool
combineList xs = combine xs == foldl1 (++) ([] : xs)

tests :: [Test]
tests =
  [ Test "1.1" "normalize is idempotent" normalizeIdempotentTest,
    Test "1.1" "normalize preserves value" normalizeValueTest,
    Test "1.1" "show preserves normalized value" showPolyTest1,
    Test "1.1" "show is consistent for equal polynomials" showPolyTest2,
    Test "1.2" "Eq works" eqPolyTest,
    Test "1.2" "Second Eq Tests" eqPolyTest2,
    Test "1.2" "Ord works" ordPolyTest,
    Test "1.3" "addition commutative" polyAddComm,
    Test "1.3" "addition associative" polyAddAssoc,
    Test "1.3" "multiplication commutative" polyMultComm,
    Test "1.3" "multiplication associative" polyMultAssoc,
    Test "1.3" "addition zero" polyAddZero,
    Test "1.3" "multiplication zero" polyMultZero,
    Test "1.3" "multiplication one" polyMultOne,
    Test "1.3" "addition with negate" polyAddNegate,
    Test "1.3" "distributive law" polyDistrib,
    Test "2.1" "monoid laws for Integer are satisfied" integerMonoidLaws,
    Test "2.1" "monoid laws for Bool are satisfied" boolMonoidLaws,
    Test "2.1" "monoid laws for [a] are satisfied" listMonoidLaws,
    Test "2.2" "normaliseVote xs doesn't contain canceling pairs" normaliseTest,
    Test "2.3" "VS xs == VS ys returns True if xs and ys represent same total vote" voteSeqEqTest,
    Test "2.3" "VS xs /= VS ys returns True if xs and ys represent different total vote" voteSeqEqTest,
    Test "2.3" "result of VoteSeq addition (binop) is normalized" voteSeqAdditionNormalized,
    Test "2.3" "result of VoteSeq addition (binop) is correct" voteSeqAdditionCorrectResult,
    Test "2.3" "x + e = e + x = x" voteSeqNeutral,
    Test "2.3" "show returns internal string representation" voteSeqShow,
    Test "2.4" "combine works correct for our Integer monoid" combineInteger,
    Test "2.4" "combine works correct for our Bool monoid" combineBool,
    Test "2.4" "combine works correct for our list monoid" combineList
  ]

boolTests :: [((String, String), Bool)]
boolTests = map (\(Test ex n t) -> ((ex, n), holds 1000 t)) tests
