module Female =
struct

let upd_fea = ref false;; (* Add features of the current literal to the cached features of the path *)
let do_nbayes = ref false;; (* Call nbayes to sort by relevance and set other weights *)

(* datai file - training data *)
let cdata = ref "";;

(* Weight of steps. 100 for all 3 is like leanCoP. *)
let optw = ref 1;;        (* weight of an optimal step *)
let singlew = ref 1;;     (* weight of an only-choice step *)
let nonoptw = ref 1;;     (* weight of a non-optimal step *)

(* NBayes params *)
let initwei = ref 2.0;;    (* initial weight of previous example number *)
let posweight = ref 2.0;;  (* weight for features shared with conjecture *)
let negweight = ref (-6.0);; (* weight for conjecture features never shared with contra *)
let invwei = ref (-0.05);;    (* weight for contra features not in conjecture *)

let args = [
  ("-cdata", Arg.String (fun s -> upd_fea := true; do_nbayes := true; cdata := s), "%s Set contrapositive training data file");
  ("-nb-initwei", Arg.Set_float initwei, "%f Set NB previous uses weight (default: 2.0)");
  ("-nb-poswei", Arg.Set_float posweight, "%f Set NB cooccurring feature weight (default: 2.0)");
  ("-nb-negwei", Arg.Set_float negweight, "%f Set NB conj-only feature weight (default: -6.0)");
  ("-nb-invwei", Arg.Set_float invwei, "%f Set NB contra-only feature weight (default: -0.05)");
  ("-step", Arg.Set_int nonoptw, "%i Set step granularity (default: 1)");
  ("-step-single", Arg.Set_int singlew, "%i Set single-step subtract (default: 1)");
  ("-step-opt", Arg.Set_int optw, "%i Set chosen-step subtract (default: 1)")
]

end


module Feats =
struct

let fea_undersubst = ref true;;  (* Check variables in the substitution when computing features *)
let fea_const      = ref true;;  (* Include consts when computing features *)
let fea_subterm    = ref false;; (* Include subterms in computed features *)
(* A feature in the path that is not in the current literal is multiplied by
   the following decay factor for each step away from current lit *)
let weaken_feature = ref 0.8;;

let args = [
  ("-feanosubst", Arg.Clear fea_undersubst, " When computing features do not descend in substitution");
  ("-feanoconst", Arg.Clear fea_const, " Do not include constant features");
  ("-feasubterm", Arg.Set fea_subterm, " Include subterm features");
  ("-feaweaken", Arg.Set_float weaken_feature, "%f Factor to multiply next feature on path (0.0--1.0, default: 0.8)")
]

end


module Lean =
struct

(* LeanCoP params *)
let conj = ref true       (* conjecture-directed: if false negative clauses have # added *)
let defcnf = ref true     (* definitional CNF. More predicates, less clauses *)
let content = ref true    (* content-based names *)
let paths = ref true
let mlproof = ref false
let shuffle = ref false

let schedule = ref 1
let depthsec = ref 0
let maxdepth = ref 7

let cut1 = ref true       (* cut after lemma step *)
let cut2 = ref true       (* cut after path step *)
let cut3 = ref true       (* cut after lit step *)

let args = [
  ("-noconj", Arg.Clear conj, " Disable conjecture-directed search");
  ("-nodefcnf" , Arg.Clear defcnf, " Disable definitional CNF");
  ("-nocontent", Arg.Clear content, " Disable content-based names");
  ("-nopaths", Arg.Clear paths, " Do not sort clauses by number of paths");
  ("-mlproof", Arg.Set mlproof, " Show proof in machine learning format");
  ("-shuffle", Arg.Set shuffle, " Shuffle matrix");
  ("-schedule", Arg.Set_int schedule, "%i Strategy schedule: 0 = single, 1 = contiCoP, 2 = FEMaLeCoP (default: 1)");
  ("-depthsec", Arg.Set_int depthsec, "%i Maximum time in seconds to spend at a depth (default: unlimited)");
  ("-maxdepth", Arg.Set_int maxdepth, "%i Maximum depth for all but last strategy (default: 7)");
  ("-nocut1", Arg.Clear cut1, " Disable cut after lemma step");
  ("-nocut2", Arg.Clear cut2, " Disable cut after reduction step");
  ("-nocut3", Arg.Clear cut3, " Disable cut after extension step")
]

end


module Trace =
struct

let trace_file = ref ""
let verbose = ref false

let args = [
  ("-trace", Arg.Set_string trace_file, "%s Output the proof trace to a file");
  ("-verbose", Arg.Set verbose, " Enable debugging messages")
]

end


module Random =
struct

let seed = ref 0

let args =
[ ("-seed", Arg.Set_int seed, "%d Random seed (default: 0)") ]
end


module Monte =
struct

module Rollout =
struct

let invprob = ref 1.
let constprob = ref 0.
let bayesprob = ref 0.

let prevdampen = ref 0.0
let prefilter = ref false

let rollout_args =
[ ("-invprob", Arg.Set_float invprob, "%f Impact of inverse clause size on probability (default: 1.0)")
; ("-constprob", Arg.Set_float constprob, "%f Constant impact on probability (default: 0.0)")
; ("-bayesprob", Arg.Set_float bayesprob, "%f Bayes impact on probability (default: 0.0)")
; ("-prevdampen", Arg.Set_float prevdampen, "%f Punishment for selecting contrapositive that was already selected on path (default: 0.0)")
; ("-prefilter", Arg.Set prefilter, " Filter extension states before calculating probability")
]

end

module Evaluation =
struct

let ldata = ref ""
let naivereward = ref 0.0
let sizereward = ref 0.
let randreward = ref 0.0
let mlreward = ref 1.0

let depth_att_max = ref 0.5

type mean_function = Min | Product | Harmonic | Geometric | Arithmetic
let read_mlmean = function
    "min" -> Min
  | "prod" -> Product
  | "harm" -> Harmonic
  | "geom" -> Geometric
  | "arit" -> Arithmetic
  | s -> failwith ("Unknown mlmean: " ^ s)
let mlmean = ref Min

let defscore = ref 0.9
let cert = ref 1.0
let certdamp = ref 1.0

let evaluation_args =
[ ("-ldata", Arg.Set_string ldata, "%s Set literal training data file")
; ("-naivereward", Arg.Set_float naivereward, "%f Set naive reward impact (default: 0.0)")
; ("-sizereward", Arg.Set_float sizereward, "%f Set literal size impact (default: 0.0)")
; ("-randreward", Arg.Set_float randreward, "%f Set random reward impact (default: 0.0)")
; ("-mlreward", Arg.Set_float mlreward, "%f Set impact of machine learning data (default: 1.0)")
; ("-depthattmax", Arg.Set_float depth_att_max, "%f Maximal depth-based attenuation for size reward (default: 0.5)")
; ("-mlmean", Arg.String (fun s -> mlmean := read_mlmean s), "%s min, prod, harm, geom, arit (default: min)")
; ("-defscore", Arg.Set_float defscore, "%f Score for previously unseen lit/cls (default: 0.9)")
; ("-cert", Arg.Set_float cert, " Influence of certainty for literal refutability (default: 1.0)")
; ("-certdamp", Arg.Set_float certdamp, "%f How much we distrust statistics (default: 1.0)")
]

end

let tradinfs = ref 0
let maxiters = ref (-1)
let dot_out = ref ""
let simdepth = ref 100
let exploration = ref 1.0
let expamp = ref 0.0
let expper = ref 50.0

type expansion_policy = First | Cut of cut_policy
and cut_policy = On_clauses | On_literals
let read_exppol = function
    "fst" -> First
  | "cutlit" -> Cut On_literals
  | "cutcla" -> Cut On_clauses
  | s -> failwith ("Unknown expansion policy: " ^ s)
let exppol = ref First

let args = Rollout.rollout_args @ Evaluation.evaluation_args @
[ ("-tradinfs", Arg.Set_int tradinfs, "%d Maximal number of traditional inferences per depth before UCT kicks in (default: 0)")
; ("-maxiters", Arg.Set_int maxiters, "%d Maximal number of UCT iterations, -1 for no limit (default: -1)")
; ("-dotout", Arg.Set_string dot_out, "%s Save Monte Carlo tree as Graphviz file")
; ("-simdepth", Arg.Set_int simdepth, "%d Set maximal depth of simulations (default: 100)")
; ("-exploration", Arg.Set_float exploration, "%f Weight of UCT exploration term (default: 1.0)")
; ("-expamp", Arg.Set_float expamp, "%f Amplitude of exploration oscillation (default: 0.0)")
; ("-expper", Arg.Set_float expper, "%f Period of exploration oscillation (default: 50.0)")
; ("-exppol", Arg.String (fun s -> exppol := read_exppol s), "%s Expansion policy: fst, cutlit, cutcla (default: fst)")
]

end


module Oracle =
struct

let n_proofs = ref (-1)

let args = [
  ("-proofs", Arg.Set_int n_proofs, "%d Number of MC proofs per literal, -1 for no limit (default: -1)")
]

end


module IO =
struct

let  infile = ref ""
let outfile = ref ""

let args = [
  ("-i", Arg.Set_string  infile, "%s Input file");
  ("-o", Arg.Set_string outfile, "%s Output file")
]

end
