open Arglean.Lean
open Arglean.Trace
open Batteries
open Term
open Mapping


module Stats =
struct

(* total number of inferences performed during proof search *)
let infer = ref 0
(* current maximal depth of proof search *)
let depth = ref 0
(* number of inferences performed until last depth change *)
let depthinfer = ref 0
(* currently running strategy *)
let strategy = ref 0

let set_depth d =
  depth := d;
  depthinfer := !infer

let reset_stats () =
  infer := 0;
  depth := 0;
  depthinfer := 0

let print_stats () =
  Format.printf "%% Inf: %i Depth: %i DInf: %i Str: %i\n%!"
    !infer !depth (!infer - !depthinfer) !strategy

end


exception Alarm
exception CleanExit

let setup_signals () =
  ignore (Sys.signal Sys.sigint  (Sys.Signal_handle (fun _ -> raise CleanExit)));
  ignore (Sys.signal Sys.sigterm (Sys.Signal_handle (fun _ -> raise CleanExit)));
  ignore (Sys.signal Sys.sigalrm (Sys.Signal_handle (fun _ -> raise Alarm)))

(* run f a for maximally i seconds (forever if i = 0) *)
let maxsec f = function
    0 -> Lazy.force f
  | i ->
    ignore (Unix.alarm i);
    try let y = Lazy.force f in ignore (Unix.alarm 0); y
    with Alarm -> None

let run_depth f d = Stats.set_depth d; maxsec (lazy (f d)) !depthsec

let iteratively_deepen f max_depth =
  LazyList.range 0 max_depth
  |> LazyList.filter_map (run_depth f)
  |> LazyList.peek

let get_schedule load_db start = function
  0 -> [lazy (iteratively_deepen start 1000)]
| 1 -> (* contiCoP schedule *)
    [ lazy (iteratively_deepen start !maxdepth)
    ; lazy (cut2 := false; cut3 := false; iteratively_deepen start !maxdepth)
    ; lazy (load_db false !defcnf; iteratively_deepen start 1000)
    ]
| 2 -> (* FEMaLeCoP schedule *)
    [ lazy (iteratively_deepen start !maxdepth)
    ; lazy (cut3 := false; maxsec (lazy (iteratively_deepen start !maxdepth)) 10)
    ; lazy (cut2 := false; maxsec (lazy (iteratively_deepen start 1000)) 10)
    ]
| _ -> failwith "unknown schedule"



let run_schedule load_db start =
  Stats.reset_stats ();
  Random.init !Arglean.Random.seed;
  load_db !conj !defcnf;
  LazyList.of_list (get_schedule load_db start !schedule) |>
  LazyList.mapi (fun i prf ->
    (*if !verbose then Format.printf "Strategy: %i\n%!" (i+1);*)
    Stats.strategy := (i+1); Lazy.force prf) |>
  LazyList.filter_map identity |> LazyList.peek

let print_status = Format.printf "%% SZS status %s\n%!"
let print_error e = print_status ("Error: " ^ e)

let show_result = function
  Some (sub, prf) -> print_status "Theorem";
    if !mlproof then Proof.Clausal.Proof.to_steps ((hash, [])) [] [] prf |> List.rev |> Proof.Clausal.ML.print_proof sub
    else Proof.Clausal.Proof.print_proof_hsh sub ((hash, []), prf)
| None -> print_status "CounterSatisfiable"

let print_error = function
  CleanExit -> print_status "Unknown"
| Failure x -> print_error x
| Parsing.Parse_error -> print_error "Parse_error"
| e -> raise e

let cut p l1 l2 =
  if p then
    lazy (match LazyList.peek l1 with
      Some x -> LazyList.Cons(x, LazyList.nil)
    | None -> LazyList.next l2)
  else
    LazyList.append l1 l2
