open Batteries
open Fof
open Term
open Mapping


let lit_hash =
  Term.rename_lit (IM.empty, 0) %> snd %> Print.string_of_lit %> Mapping.md5s

let form_hash = Fof.rename_form true %> Print.string_of_form %> Mapping.md5s


module Clausal =
struct

(* (lit-arguments, rest-clause, vars, contrapositive hash) *)
type contrapositive = (iterm list * lit list * int * int)

(* contrapositive <-> number mappings *)
let contr_no : (string, int) Hashtbl.t = Hashtbl.create 100
and no_contr : (int, string) Hashtbl.t = Hashtbl.create 100
and no_rcont : (int, lit * lit list) Hashtbl.t = Hashtbl.create 100

let rec find_free_contr n =
  if Hashtbl.mem no_contr n then find_free_contr (n + 1) else n

let form_of_contra (lit, rest) =
  List.fold_right (fun a b -> Disj (form_of_lit (negate_lit a), b)) rest (form_of_lit lit)

let string_of_contr = form_of_contra %> rename_form true %> Print.string_of_form

let contr_number lit rest =
  let name = string_of_contr (lit, rest) in
  try Hashtbl.find contr_no name with Not_found ->
    let cno = find_free_contr (Mapping.md5s name) in
    (*Format.printf "Contr: %d, %s\n%!" cno name;*)
    Hashtbl.add contr_no name cno;
    Hashtbl.add no_contr cno name;
    Hashtbl.add no_rcont cno (lit, rest);
    cno


(* run function for every element and all other elements of the list *)
let rec iter_rest acc f = function
    [] -> ()
  | h :: t -> f h (List.rev_append acc t); iter_rest (h :: acc) f t;;

let iter_rest_nohash acc f =
  iter_rest acc (fun h rest -> f h (List.filter ((<>) hash_lit) rest))

let cl2predb predb cl =
  let max_var = Matrix.Clausal.clause_max_var cl in
  iter_rest_nohash [] (fun (p, tl) rest -> Hashtbl.add predb p
    (tl, rest, max_var, contr_number (p, tl) rest)) (List.rev cl)

let axs2predb axs =
  let predb = Hashtbl.create 100 in
  List.iter (cl2predb predb) axs; predb


module ClassifierDb =
struct

open Features

(* contrapositive, (card, sum_te_ftr, ftr_sum_te) *)
type db_entry = contrapositive * (float * float * float FM.t)

(* for every predicate, store list of possible contrapositives *)
let db : (int, db_entry list) Hashtbl.t = Hashtbl.create 10017


let contrapositive_ml classifier feats (_, _, _, cn) =
  let relevant (k,l) _ = FS.mem (k,l) feats || FS.mem (-k,l) feats in
  let (card, sum_te_ftr, ftr_sum_te) = FClassifier.get_lbl_data classifier cn in
  (float_of_int card, sum_te_ftr, FM.filter relevant ftr_sum_te)

let clauses_features = List.fold_left (List.fold_left (lit_feats [])) FS.empty

let axioms2db classifier axs =
  let feats = clauses_features ([(hash, [])] :: axs) in
  let add_ml c = (c, contrapositive_ml classifier feats c) in
  let predb = axs2predb axs in
  Hashtbl.clear db;
  Hashtbl.iter (fun k v -> Hashtbl.modify_def [] k (List.cons (add_ml v)) db) predb

let db_entries sub neglit =
  try Hashtbl.find db (fst neglit) with Not_found -> []

end


module ListDb =
struct

let db : (int, contrapositive list) Hashtbl.t = Hashtbl.create 10017

let axioms2db axs =
  let predb = axs2predb axs in
  Hashtbl.clear db;
  Hashtbl.iter (fun k v -> Hashtbl.modify_def [] k (List.cons v) db) predb

let db_entries sub neglit =
  try Hashtbl.find db (fst neglit) with Not_found -> []

end


module LazyDb =
struct

let db : (int, contrapositive LazyList.t) Hashtbl.t =
  Hashtbl.create 10017

let axioms2db axs =
  let predb = axs2predb axs in
  Hashtbl.clear db;
  Hashtbl.iter (fun k v -> Hashtbl.modify_def LazyList.nil k (LazyList.cons v) db) predb

let db_entries sub lit = lazy (
  try Hashtbl.find db (fst lit) |> LazyList.next with Not_found -> LazyList.Nil)

end

end



module Nonclausal =
struct

open Extclause

(* position of the literal in matrix, literal arguments, greatest variable *)
type 'v contrapositive = 'v clause_ext * 'v term list * int

let db : (int, int contrapositive Lazylist.t) Hashtbl.t = Hashtbl.create 10000

let insert_db db (((p, a), off), clext) =
  let v = clext, a, off in
  try Hashtbl.find db p |> Lazylist.cons v |> Hashtbl.replace db p
  with Not_found -> Hashtbl.add db p (Lazylist.cons v Lazylist.nil)

let matrix2db m =
  let predb = assert_matrix m in
  Hashtbl.clear db;
  List.iter (insert_db db) (List.rev predb)

let db_entries lit () =
  try Hashtbl.find db (fst lit) |> Lazylist.next with Not_found -> Lazylist.Nil

end
