open Batteries
open Mapping
open Term
open Fof_parse
open Arglean.Feats
open Arglean.Female

module FeatureOrdered = struct type t = lit let compare = compare end

module FS = Set.Make(FeatureOrdered)
module FM = Map.Make(FeatureOrdered)
module FClassifier = Classifier.Make (FeatureOrdered)


let classifier : (int FClassifier.t) ref = ref (FClassifier.empty ())


let rec term_feats sub acc = function
  | V n -> if !fea_undersubst then (try let t = List.assoc n sub in term_feats sub acc t with Not_found -> acc) else acc
  | A (f, []) -> if !fea_const || !fea_subterm then FS.add (f, []) acc else acc
  | A (f, l) ->
     let acc = List.fold_left (term_feats sub) acc l in
     let acc = if !fea_const then FS.add (f, []) acc else acc in
     if !fea_subterm then FS.add (f, List.map normalize l) acc else acc

let lit_feats sub acc (p, l) = FS.add (p, []) (List.fold_left (term_feats sub) acc l)

let update_ftrw sub (sf, weight) tm =
  let fea = lit_feats sub FS.empty (Fof.lit_of_form tm) in
  let ret = FS.fold (fun f sf ->
    FM.add f (max weight (try FM.find f sf with Not_found -> 0.)) sf) fea sf in
  ret, weight *. !weaken_feature

let path_features sub = function
    [] -> FM.empty
  | h::_ as l ->
      if !weaken_feature > 0.0
      then fst (List.fold_left (update_ftrw sub) (FM.empty, 1.0) l)
      else fst (update_ftrw sub (FM.empty, 1.0) h)

let mult_map map = function
    0. -> FM.empty
  | 1. -> map
  | c -> FM.map (Float.mul c) map

let update_features sub lit =
  let ftrs = lit_feats sub FS.empty lit
  and weaken_features sf = mult_map sf !weaken_feature in
  weaken_features %> FS.fold (fun ftr sf -> FM.add ftr 1. sf) ftrs


let relevance feats (tfreq, _, sfreq) =
  let fl idf w = idf *. !negweight
  and fi idf (w, sf) = idf *. log (!posweight *. sf /. tfreq)
  and fr idf sf = idf *. !invwei *. log (1. -. sf /. (tfreq +. 1.)) in
  FClassifier.relevance (FClassifier.get_idf !classifier) fl fi fr feats sfreq
  +. !initwei *. log (tfreq +. 1.)

let lbl_relevance lbl_data =
  FClassifier.lbl_relevance !classifier lbl_data
let ftr_relevance feats lbl_data =
  FClassifier.ftr_relevance !classifier feats lbl_data

let treat_scores = function
    [] -> []
  | [(x, _)] -> [(x, !singlew)]
  | (x, _) :: t -> (x, !optw) :: List.map (fun (x, _) -> (x, !nonoptw)) t

let relevancel fea =
  if !do_nbayes
  then List.rev_map (fun (x, freqs) -> x, relevance fea freqs) %>
       List.stable_sort (fun (_, a) (_, b) -> compare b a) %>
       treat_scores
  else List.map (fun (x, _) -> (x, !nonoptw))
