open Batteries
open BatFixes
open LazyList
open Option.Infix

open Arglean.Lean
open Arglean.Trace
open Cnf
open Common
open Database.Clausal.LazyDb
open Mapping
open Term
open Print
open Proof.Clausal.Proof

module Subst = Substitution.Substoff (Substitution.Substarray)

let rec prove_lit sub (path, lem, lim) lit =
  if !verbose then Format.printf "Lit: %s\n%!" (string_of_lit lit);
  if !verbose then Format.printf "Path: %s\n%!" (string_of_lits (LazyList.to_list path));
  if !verbose then Format.printf "Lemmas: %s\n%!" (string_of_lits lem);
  let neglit = negate_lit lit in
  let lemmas =
    if List.exists (Subst.eq sub lit) lem then (if !verbose then Format.printf "lemma\n%!"; cons (sub, lem, Lemma) nil) else nil
  and reductions = LazyList.filter_map
    (fun p -> if !verbose then Format.printf "Reduction try %s\n%!" (string_of_lit p);
      Subst.unify sub neglit p >>=
      (fun sub1 -> if !verbose then Format.printf "Reduction works\n%!";
        Some (sub1, lem, Reduction))
    ) path
  and extensions = db_entries sub neglit
    |> LazyList.map (fun ((_, _, vars, hsh) as contra) ->
      if !verbose then Format.printf "Extension try %s (for lit %s, lim %d)\n%!" (Hashtbl.find Database.Clausal.no_contr hsh) (string_of_lit lit) lim;
      if lim <= 0 && vars > 0 then nil
      else match Subst.unify_rename sub (snd lit) contra with
        Some (sub1, cla1) ->
          if !verbose then Format.printf "Extension works\n%!";
          incr Stats.infer;
          prove_clause sub1 (lit ^:^ path, lem, lim - 1) (cla1, hsh)
          |> LazyList.map (fun (sub2, prfs) -> (sub2, lit :: lem, Extension (hsh, prfs)))
      | None -> nil)
    |> LazyList.concat in
  cut !cut1 lemmas (cut !cut2 reductions (cut !cut3 extensions nil))
and prove_clause sub (path, lem, lim) (cl, cl_hsh) = match cl with
    lit :: lits ->
    if (List.exists (fun x -> LazyList.exists (Subst.eq sub x) path)) cl then (if !verbose then Format.printf "regularity\n%!"; nil)
    else
    prove_lit sub (path, lem, lim) lit
    |> Litdata.trace_proofs lit cl_hsh
    |> LazyList.map
       (fun (sub1, lem1, prf1) -> prove_clause sub1 (path, lem1, lim) (lits, cl_hsh) |> LazyList.map
       (fun (sub2, prfs) -> (sub2, (lit, prf1) :: prfs)))
    |> LazyList.concat
  | [] -> cons (sub, []) nil


let start lim =
  if !verbose then Format.printf "Start %d\n%!" lim;
  Hashtbl.clear Litdata.cl_out;
  prove_lit (Subst.empty 1000000, 0) (nil, [], lim) (hash, []) |> LazyList.peek
  >>= (fun (sub, _, prfs) -> Some (fst sub |> Subst.to_list, prfs))

let leancop file =
  try
    let load_db conj def = axioms2db (file_mat conj def file) in
    run_schedule load_db start |> Option.map Litdata.write_trace |> show_result;
    Stats.print_stats ()
  with e -> print_error e; Stats.print_stats ()


let _ =
  setup_signals ();

  let tosolve = ref []
  and speclist = Arg.align Arglean.(Trace.args @ Lean.args)
  and usage = "Usage: lazycop [options] <file.p>\nAvailable options are:" in
  Arg.parse speclist (fun s -> tosolve := s :: !tosolve) usage;

  if !tosolve = [] then Arg.usage speclist usage
  else List.iter leancop (List.rev !tosolve)
