open Batteries
open BatFixes
open Option.Infix
open Common
open Fof
open Term
open Matrix.Nonclausal
open Lazylist
open Arglean.Trace
open Arglean.Lean
open Print
open Proof.Nonclausal
open Extclause

module Subst = Substitution.Substoff (Substitution.Substarray)

let verbosenc = false

let file_mat conj = let open Preprocess in
  Fof_lexer.file %> List.rev %>
  split_goal %> (if conj then add_hash else identity) %> combine_goal %>
  map_form_vars Mapping.var %>
  Equality.equal_axioms Mapping.eqn %>
  neg %>
  unfold_equiv true %> miniscope %> nnf %> rename_form true %> skolem !content %>
  (*tap (List.length %> Format.printf "Number of formulae: %d\n%!") %>
  tap (List.iter (string_of_form %> Format.printf "Formula: %s\n%!")) %>*)
  (*tap (string_of_form %> Format.printf "Formula: %s\n%!") %>*)
  matrix_of_form %>
  (if !paths then matrix_paths else identity) %>
  index_matrix


let unify_vars (sub, off) l1 l2 =
  let v x = V x in
  try Some (Subst.unify_tms sub (List.map v l1) (List.map v l2), off)
  with Subst.Unify -> None

let unify_rename (sub, off) args1 (clext, args2, vars) =
  try Some (
    if vars = 0 then ((Subst.unify_tms sub args1 args2, off), clext)
    else
      (Subst.unify_tms_off off sub args1 args2, off + vars),
      offset_clause_ext off clext)
  with Subst.Unify -> None


(* epsilon operator on lists *)
let pick f l =
  let rec go acc = function
    x :: xs -> begin match f x with
      Some y -> (y, fun z -> List.rev_append acc (z::xs))
    | None -> go (x::acc) xs end
  | [] -> failwith "pick: no element found" in
  go [] l

let nth_clause i ((((j, k), v), _) as c) = if i = j then Some c else None
let nth_matrix i = function
    Lit _ -> None
  | Mat ((j, k), _ as m) -> if i = j then Some m else None

let rec prove_ec k sub ((i, v), _, lmext as clext) mi pi =
  let ((((i, k1), v1), cla1), miAB) = pick (nth_clause i) mi in
  let alt1 () =
    match lmext with Matext (j, _, clext') when List.mem (j, k1) pi ->
      begin match unify_vars sub v v1 with None -> Nil | Some sub2 ->
        let (((j, k1'), mi2), claDE) = pick (nth_matrix j) cla1 in
        assert (k1 = k1');
        prove_ec k sub2 clext' mi2 pi |> Lazylist.map
          (fun (sub3, (prefix, postfix), claB1, mi3) ->
            (sub3, (((i, v1), j) :: prefix, postfix), claB1, miAB (((i, k1), v1), claDE (Mat ((j, k1), mi3))))
          ) |> Lazylist.next
      end
    | _ -> Nil
  and alt2 () =
    if List.mem (i, k1) pi && v = [] then Nil
    else
      let (claB, claC) = claBC_of_litmat_ext lmext in
      let index cla = (((i, k), v), copy_clause k cla) in
      Cons ((sub, ([], clext), index claB, miAB (index claC)), nil) in
  Lazylist.append alt1 alt2

let cut p l1 l2 =
  if p then
    fun () -> match Lazylist.peek l1 with
      Some x -> Lazylist.Cons(x, Lazylist.nil)
    | None -> Lazylist.next l2
  else
    Lazylist.append l1 l2


let pred_of_lit (p, _) = if p < 0 then -p, [] else p, []

let rec prove_lit sub (mi, path, pi, lem, lim) lit =
  (*if verbosenc then Format.printf "%s\n%!" (string_of_lit (pred_of_lit lit));*)
  if !verbose then Format.printf "Lit: %s\n%!" (string_of_lit (Subst.inst_lit (fst sub) lit));
  if !verbose then Format.printf "Path: %s\n%!" (string_of_lits (Lazylist.to_list path));
  if !verbose then Format.printf "Lemmas: %s\n%!" (string_of_lits lem);
  let neglit = negate_lit lit
  and k = lazy (List.length pi) in
  let lemmas =
    if List.exists (Subst.eq sub lit) lem then (if !verbose then Format.printf "lemma\n%!"; cons (sub, lem, Lemma) nil) else nil
  and reductions = Lazylist.filter_map
    (fun p -> if !verbose then Format.printf "Reduction try %s\n%!" (string_of_lit p);
      match Subst.unify sub neglit p with
        Some sub1 -> if !verbose then Format.printf "Reduction works\n%!";
        Some (sub1, lem, Reduction)
      | None -> None)
    path
  and extensions = Database.Nonclausal.db_entries neglit
    |> Lazylist.map (fun ((_, _, vars) as contra) ->
      if !verbose then Format.printf "Extension try (for lit %s, lim %d, vars %d)\n%!" (string_of_lit lit) lim vars;
      if lim < 0 && vars > 0 then nil
      else match unify_rename sub (snd lit) contra with
        Some (sub1, clext) ->
          if !verbose then Format.printf "Extension works (for lit %s, lim %d)\n%!" (string_of_lit lit) lim;
          if !verbose then Format.printf "Old/new offsets: %d/%d\n%!" (snd sub) (snd sub1);
          incr Stats.infer;
          prove_ec (Lazy.force k) sub1 clext mi pi
          |> Lazylist.map (fun (sub2, position, ((i, v), claB1), mi1) ->
            if !verbose then Format.printf "Extension clause %s found\n%!" (string_of_nclause claB1);
            prove_clause sub2 (mi1, Lazylist.cons lit path, i::pi, lem, lim - 1) claB1
            |> Lazylist.map (fun (sub3, prfs) ->
(*
              if verbosenc then Format.printf "Lit__: %s\n%!" (string_of_lit (inst_lit (fst sub3) lit));
              if verbosenc then Format.printf "ClaB1: %s\n%!" (string_of_nclause claB1);
*)
              (sub3, lit :: lem, Extension (Lit neglit :: claB1, position, prfs))))
          |> Lazylist.concat
      | None -> nil)
    |> Lazylist.concat in
  cut (Subst.ground_lit (fst sub) lit)
  (cut !cut1 lemmas (cut !cut2 reductions (cut !cut3 extensions nil))) nil
  |> Lazylist.map (fun (sub1, lem1, prf1) -> (sub1, lem1, (Lit lit, prf1)))
(* decomposition rule *)
and prove_mat sub (mi, path, pi, lem, lim) (j, mat1) =
  if !verbose then Format.printf "prove_mat\n%!";
  Lazylist.of_list mat1 |> Lazylist.map
    (fun ((i, _), cla1) ->
      if !verbose then Format.printf "Decomposition chose: %s\n%!" (string_of_nclause cla1);
      prove_clause sub (mi, path, i::j::pi, lem, lim) cla1
      |> Lazylist.map (fun (sub1, prf1) -> (sub1, lem, (Mat (j, mat1), Decomposition (cla1, prf1))))
    ) |> Lazylist.concat
and prove_litmat sub st = map_litmat (prove_lit sub st) (prove_mat sub st)
and prove_clause sub (mi, path, pi, lem, lim) = function
  lm :: cla ->
    if (List.exists (fun x -> Lazylist.exists (Subst.eq sub x) path)) (clause_lits cla)
    then (if !verbose then Format.printf "regularity\n%!"; nil)
    else prove_litmat sub (mi, path, pi, lem, lim) lm |> Lazylist.map
      (fun (sub1, lem1, prf1) ->
        prove_clause sub1 (mi, path, pi, lem1, lim) cla
        |> Lazylist.map (fun (sub2, prf2) -> sub2, prf1 :: prf2)
      ) |> Lazylist.concat
  (* axiom *)
  | [] -> if !verbose then Format.printf "axiom\n%!"; cons (sub, []) nil


let start mat lim =
  if !verbose || verbosenc then Format.printf "Start %d\n%!" (lim+1);
  let mati = copy_matrix 0 mat in
  let sub0 = Subst.empty 1000000 in
  Lazylist.of_list mati |> Lazylist.filter_map
    (fun (((i, _), v), cla) ->
      clause_positive cla >>= (fun cla1 -> let off = clause_offset cla1 in
      if verbosenc then Format.printf "Positive clause\n%!";
      if !verbose then Format.printf "New start clause: %s\n%!" (string_of_nclause cla1);
      prove_clause (sub0, off) (mati, nil, [i, 0], [], lim) cla1 |>
      Lazylist.peek |>
      Option.map (fun (sub, prfs) -> sub, Decomposition (cla1, prfs)))
    ) |> Lazylist.peek >>= (fun ((sub, off), prf) -> Some (Subst.to_list sub, (Mat ((-1, 0), mati), prf)))

let show_result = function
  Some (sub, prf) -> print_status "Theorem";
    Proof.Nonclausal.print_proof_ncl sub prf
| None -> print_status "CounterSatisfiable"

let nanocop file =
  try
    let mat = file_mat false file in
    if !verbose then (
      Format.printf "Matrix:\n%!";
      print_matrix_i mat;
      Format.printf "Loading %s successful.\n%!" file);
    let load_db conj def = Database.Nonclausal.matrix2db mat in
    if !verbose then Format.printf "Matrix loaded.\n%!";
    run_schedule load_db (start mat) |> show_result;
    Stats.print_stats ()
  with e -> print_error e; Stats.print_stats ()

let _ =
  setup_signals ();

  let tosolve = ref []
  and speclist = Arg.align Arglean.(Trace.args @ Lean.args)
  and usage = "Usage: lazycop [options] <file.p>\nAvailable options are:" in
  Arg.parse speclist (fun s -> tosolve := s :: !tosolve) usage;

  if !tosolve = [] then Arg.usage speclist usage
  else List.iter nanocop (List.rev !tosolve)
