%% File: nanocop.pl  -  Version: 1.0  -  Date: 16 January 2016
%%
%% Purpose: nanoCoP: A Non-clausal Connection Prover
%%
%% Author:  Jens Otten
%%
%% Usage:   prove(F,P).  % where F is a first-order formula, e.g.
%%                       %  F=((p,all X:(p=>q(X)))=>all Y:q(Y))
%%                       %  and P is the returned connection proof

:- lib(iso).  % load library for ISO compatibility
:- set_flag(occur_check,on).  % global occur check on

% Inform the interpreter that the definition of the predicate(s)
% may change during execution (using assert/1 and/or retract/1).
:- dynamic(pathlim/0), dynamic(lit/4).

% definitions of logical connectives and quantifiers

:- op(1130,xfy,<=>). :- op(1110,xfy,=>). :- op(500, fy,'~').
:- op( 500, fy,all). :- op( 500, fy,ex). :- op(500,xfy,:).

% -----------------------------------------------------------------
% prove(F,Proof) - prove formula F

% prove/2
prove(F,Proof) :- prove2(F,[cut,comp(6)],Proof).

% prove/3
prove2(F,Set,Proof) :-
    bmatrix(F,Mat), retract_all(lit(_,_,_,_)), assert_matrix(Mat),
    prove(Mat,1,Set,Proof).

verbose_l :- false.
printl(X) :- verbose_l -> print(X), nl ; true.

% prove/4
% start rule
prove(Mat,PathLim,Set,[(I^0)^V:Cla1|Proof]) :-
    (verbose_l -> print('Start '), print(PathLim), nl ; true),
    %printl('Set' : Set),
    %print(Mat), nl,
    % get the first copy of a matrix clause
    member((I^0)^V:Cla,Mat),
    % delete all negative clauses in Cla and store in Cla1
    %positiveC(Cla,Cla1),
    Cla1=Cla,
    % Cla1 must be non-empty
    % (! usually is cut, but represents the empty clause from positiveC)
    Cla1\=!,
    printl('Positive clause'),
    printp('New start clause': Cla1),
    prove(Cla1,Mat,[],[I^0],PathLim,[],Set,Proof).

% iterative deepening
prove(Mat,PathLim,Set,Proof) :-
    %print('PathLim': PathLim), nl,
    retract(pathlim) ->
    ( member(comp(PathLim),Set) -> prove(Mat,1,[],Proof) ;
      PathLim1 is PathLim+1, prove(Mat,PathLim1,Set,Proof) ) ;
    member(comp(_),Set) -> prove(Mat,1,[],Proof).


printn(X) :- print(X), nl.

verbose_p :- false.
printp(X) :- verbose_p -> printn(X); true.

pred_of_lit(L, P) :-
  (L = -X -> L1 = X ; L1 = L),
  L1=..[P|_].


% prove/8
% axiom
prove([],_,_,_,_,_,_,[]).

% prove/8
% decomposition rule
prove([J:Mat1|Cla],MI,Path,PI,PathLim,Lem,Set,Proof) :- !,
    % get a clause Cla1 with index I from Mat1 with index/variant J
    member(I^_:Cla1,Mat1),
    printp('Decomposition chose': Cla1),
    % prove Cla1, followed by remaining clause, and append proofs
    % [I,J|PI] is equivalent to I :: J :: PI in ML
    % PI contains indices of clauses and matrices that contain
    % literals on the path
    prove(Cla1,MI,Path,[I,J|PI],PathLim,Lem,Set,Proof1),
    prove(Cla,MI,Path,PI,PathLim,Lem,Set,Proof2),
    append(Proof1,Proof2,Proof).

% reduction and extension rules
prove([Lit|Cla],MI,Path,PI,PathLim,Lem,Set,Proof) :-
    %printp('MI': MI),
    printp('Lit': Lit),
    printp('Cla': Cla),
    printp('Path': Path),
    printp('Lem': Lem),
    Proof=[[I^V:[NegLit|ClaB1]|Proof1]|Proof2],
    % Lit == LitV iff Lit does not contain variables; used below
    copy_term(Lit,LitV),
    pred_of_lit(Lit, P), printl(P),

    % there should not be any shared literals between path and the matrix
    \+ (member(LitC,[Lit|Cla]), member(LitP,Path), LitC==LitP),
    (-NegLit=Lit;-Lit=NegLit) ->
       % implicit cut?
         % lemma rule
       ( member(LitL,Lem), Lit==LitL, ClaB1=[], Proof1=[], printp('lemma')
         ;
         % reduction rule
         member(NegL,Path), unify_with_occurs_check(NegL,NegLit),
         ClaB1=[], Proof1=[], printp('reduction')
         ;
         % extension rule
         lit(NegLit,ClaB,Cla1,Grnd1),
         %print('ClaB for '), print(Lit), print(': '), print(ClaB), nl,
         ( Grnd1=g -> true ; length(Path,K), K<PathLim -> true ;
           \+ pathlim -> assert(pathlim), fail ),
         printp('ClaB': ClaB),
         %printp('Cla1': Cla1),
         prove_ec(ClaB,Cla1,MI,PI,I^V:ClaB1,MI1),
         printp('ClaB1': ClaB1),
         prove(ClaB1,MI1,[Lit|Path],[I|PI],PathLim,Lem,Set,Proof1)
         %,print('Lit__': Lit), nl,
         %print('ClaB1':ClaB1), nl
       ),
       % do not try other options above if cut set or literal contains no vars
       ( (member(cut,Set);Lit==LitV) -> ! ; true ),
       prove(Cla,MI,Path,PI,PathLim,[Lit|Lem],Set,Proof2).

verbose_ec :- false.
printec(X) :- verbose_ec -> printn(X); true.

% extension clause (e-clause)
prove_ec((I^K)^V:ClaB,IV:Cla,MI,PI,ClaB1,MI1) :-
    % demand that matrix MI contains a clause with index I, and
    % that it is preceded by MIA and succeded by MIB
    append(MIA,[(I^K1)^V1:Cla1|MIB],MI),
    % set copy index to length of path
    length(PI,K),
    (
      printec('length(PI)': K),
      printec('ClaB': ClaB),
      printec('Cla1': Cla1),
      % ClaB starts with a matrix entry that has only a single clause, ClaB2
      ClaB=[J^K:[ClaB2]|_],
      printec('Option 1'),
      printec('J': J),
      printec('PI': PI),
      member(J^K1,PI),
      %printec('MI before occurs': MI),
      printec('V, V1 before unify': V: V1),
      unify_with_occurs_check(V,V1),
      printec('MI after  occurs': MI),
      % the first clause of the first matrix in Cla is Cla2
      Cla=[_:[Cla2|_]|_],
      append(ClaD,[J^K1:MI2|ClaE],Cla1),
      prove_ec(ClaB2,Cla2,MI2,PI,ClaB1,MI3),
      append(ClaD,[J^K1:MI3|ClaE],Cla3),
      append(MIA,[(I^K1)^V1:Cla3|MIB],MI1)
      ;
      printec('Option 2'),
      printec('IV:Cla': IV:Cla),
      (member(I^K1,PI) -> printec('V,V1 before': (V, V1)); true),
      (\+member(I^K1,PI); V\==V1; V\=[]) ->
      % it seems that \== does not change V/V1, even if they contain variables
      % that unify -- see e.g. output from PUZ001+2.p
      % (that also seems to be only nonclausal PUZ problem where member(I^K1,PI)
      % becomes true)
      (member(I^K1,PI) -> printec('V,V1 after ': (V, V1)); true),
      ClaB1=(I^K)^V:ClaB, append(MIA,[IV:Cla|MIB],MI1)
    ).

% -----------------------------------------------------------------
% assert_matrix(Matrix) - write matrix into Prolog's database

verbose_ac :- false.
verbose_m :- false.
printac(X) :- verbose_ac -> printn(X); true.

print_lits :-
    lit(Lit, ClaB, ClaC, Grnd),
    print([Lit, '***', ClaB, '***', ClaC, '***', Grnd]), nl,
    fail
  ; true.

assert_matrix(M) :-
    member(IV:C,M),
    printac('assert_matrix'),
    assert_clauses(C,IV:ClaB,ClaB,IV:ClaC,ClaC).
% because the first assert_matrix will always fail in the end
% (producing literal entries as side-effect), make a second
% that always succeeds which is called when matrix was processed
assert_matrix(M) :-
  verbose_m -> (
  print('Lit tuples:\n'), print_lits, print('End Lit tuples.\n'),
  print('Matrix:\n'), print(M), print('\nEnd Matrix.\n')); true.

assert_clauses(C,ClaB,ClaB1,ClaC,ClaC1) :- !,
    printac('C': C),
    printac('ClaB': ClaB), printac('ClaB1': ClaB1),
    printac('ClaC': ClaC), printac('ClaC1': ClaC1),
    append(ClaD,[M|ClaE],C),
    printac('getting a clause element'),
    ( M=J:Mat -> printac('Treating matrix': Mat),
                 append(MatA,[IV:Cla|MatB],Mat),
                 printac('Picking clause': (IV:Cla)),

                 % set ClaB1, and indirectly also ClaB
                 append([J:[IV:ClaB2]|ClaD],ClaE,ClaB1),
                 % make Mat1 with existing MatA/MatB and fresh clause ClaC2
                 append([IV:ClaC2|MatA],MatB,Mat1),
                 % set ClaC1, and indirectly also ClaC
                 append([J:Mat1|ClaD],ClaE,ClaC1),
                 assert_clauses(Cla,ClaB,ClaB2,ClaC,ClaC2)
               ; % if the matrix M is actually a literal
                 printac('literal detected'),
                 append(ClaD,ClaE,ClaB1), ClaC1=C,
                 printac('Literal C' : C),
                 (ground(C) -> Grnd=g ; Grnd=n),
                 printac('Asserting': (M,ClaB,ClaC,Grnd)),
                 assert(lit(M,ClaB,ClaC,Grnd)), fail ).

% -----------------------------------------------------------------
% bmatrix(Formula,Matrix) - generate indexed matrix

verbose_bm :- false.
printbm(X) :- verbose_bm -> printn(X); true.

% polarity 0 means positive, polarity 1 means negative (~)
% FV contains the universal quantifier prefix
% FreeV seems to be modified only in beta case
% I, I1, K are used to index and identify copies

bmatrix(F,M) :- univar(F,[],F1), bmatrix(F1,0,_:M,[],[],_,1,_,_).

bmatrix((F1<=>F2),Pol,M,FreeV,FV,Paths,I,I1,K) :- !,
    bmatrix(((F1=>F2),(F2=>F1)),Pol,M,FreeV,FV,Paths,I,I1,K).

bmatrix((~F),Pol,M,FreeV,FV,Paths,I,I1,K) :- !,
    Pol1 is (1-Pol), bmatrix(F,Pol1,M,FreeV,FV,Paths,I,I1,K).

% universal quantifier
bmatrix(F,Pol,M,FreeV,FV,Paths,I,I1,K) :-
    F=..[C,X:F1], bma(uni,C,Pol), !,
    printbm('Universal': F),
    bmatrix(F1,Pol,M,FreeV,[X|FV],Paths,I,I1,K).

% existential quantifier
bmatrix(F,Pol,M,FreeV,FV,Paths,I,I1,K) :-
    F=..[C,X:F1], bma(exist,C,Pol), !,
    printbm('Existential': F),
    append(FreeV,FV,FreeV1), I2 is I+1,
    printbm('FreeV, FV, FreeV1': (FreeV, FV, FreeV1)),
    copy_term((X,F1,FreeV1),((I^FreeV1),F2,FreeV1)),
    bmatrix(F2,Pol,M,FreeV,FV,Paths,I2,I1,K).

bmatrix(F,Pol,J^K:M3,FreeV,FV,Paths,I,I1,K) :-
    F=..[C,F1,F2], bma(alpha,C,Pol,Pol1,Pol2), !,
    bmatrix(F1,Pol1,J^K:M1,FreeV,FV,Paths1,I,I2,K),
    bmatrix(F2,Pol2,_:M2,FreeV,FV,Paths2,I2,I1,K),
    Paths is Paths1*Paths2,
    %(Paths1>Paths2 -> append(M2,M1,M3) ; append(M1,M2,M3)).
    append(M1,M2,M3).

bmatrix(F,Pol,I^K:[(I2^K)^FV1:C3],FreeV,FV,Paths,I,I1,K) :-
    F=..[C,F1,F2], bma(beta,C,Pol,Pol1,Pol2), !,
    ( FV=[] -> FV1=FV, F3=F1, F4=F2 ;
      copy_term((FV,F1,F2,FreeV),(FV1,F3,F4,FreeV)) ),
    append(FreeV,FV1,FreeV1),  I2 is I+1, I3 is I+2,
    bmatrix(F3,Pol1,M1,FreeV1,[],Paths1,I3,I4,K),
    bmatrix(F4,Pol2,M2,FreeV1,[],Paths2,I4,I1,K),
    Paths is Paths1+Paths2,
    ( (M1=_:[_^[]:C1];[M1]=C1), (M2=_:[_^[]:C2];[M2]=C2) ->
      %(Paths1>Paths2 -> append(C2,C1,C3) ; append(C1,C2,C3)) ).
      append(C1,C2,C3) ).

bmatrix(A,0,I^K:[(I2^K)^FV1:[A1]],FreeV,FV,1,I,I1,K)  :-
    copy_term((FV,A,FreeV),(FV1,A1,FreeV)), I2 is I+1, I1 is I+2.
bmatrix(A,1,I^K:[(I2^K)^FV1:[-A1]],FreeV,FV,1,I,I1,K) :-
    copy_term((FV,A,FreeV),(FV1,A1,FreeV)), I2 is I+1, I1 is I+2.

bma(alpha,',',1,1,1). bma(alpha,(;),0,0,0). bma(alpha,(=>),0,1,0).
bma(beta,',',0,0,0).  bma(beta,(;),1,1,1).  bma(beta,(=>),1,0,1).
bma(exist,all,0). bma(exist,ex,1). bma(uni,all,1). bma(uni,ex,0).

% -----------------------------------------------------------------
% positiveC(Clause,ClausePos) - generate positive start clause

positiveC([],[]).
positiveC([M|C],C1) :-
    ( M=I:M1 -> positiveM(M1,M2), M3=I:M2 ; (-_=M->M3=_:[];M3=M) ),
    ( M3=_:[] -> C1=! ; positiveC(C,C2), (C2=!->C1=!;C1=[M3|C2]) ).
positiveM([],[]).
positiveM([I:C|M],M1) :-
    positiveC(C,C1), (C1=! -> M1=M2;M1=[I:C1|M2]), positiveM(M,M2).

% -----------------------------
% create unique variable names

univar(X,_,X)  :- (atomic(X);var(X);X==[[]]), !.
univar(F,Q,F1) :-
    F=..[A,B|T], ( (A=ex;A=all) -> B=X:C, delete2(Q,X,Q1),
    copy_term((X,C,Q1),(Y,D,Q1)), univar(D,[Y|Q],E), F1=..[A,Y:E] ;
    univar(B,Q,B1), univar(T,Q,T1), F1=..[A,B1|T1] ).

% -----------------------------
% delete variable from list

delete2([],_,[]).
delete2([X|T],Y,T1) :- X==Y, !, delete2(T,Y,T1).
delete2([X|T],Y,[X|T1]) :- delete2(T,Y,T1).

