open Batteries
open Format
open Term
open Mapping
open Fof
open Matrix.Nonclausal

let pp_interleave sep fn f =
  List.map (fun x () -> fn f x) %>
  List.interleave (fun () -> sep f) %>
  List.iter (fun f -> f ())

let pp_iter sep = pp_interleave (fun f -> pp_print_string f sep)

let pp_iter_sp sep fn =
  let sep f = pp_print_char f ' '; pp_print_string f sep; pp_print_space f () in
  pp_interleave sep fn

let pp_enclose start stop fn f x =
  pp_print_string f start; fn f x; pp_print_string f stop

let pp_enclose_cut start stop fn =
  pp_enclose start stop (fun f x -> fn f x; pp_print_cut f ())

let pp_enum sep start stop fn =
  pp_enclose start stop (pp_iter sep fn)

let pp_print_pair sep fn f (x, y) =
  fn f x; pp_print_string f sep; fn f y

let pp_with_box i fn f x =
  pp_open_box f i; fn f x; pp_close_box f ()

let pp_nl fn f x = fn f x; pp_print_newline f ()

let pp_print_null f x = ()

let print_to_string printer x =
  let buf = Buffer.create 10 in
  let fmt = formatter_of_buffer buf in
  pp_set_max_boxes fmt 1000;
  printer fmt x;
  pp_print_flush fmt ();
  let s = Buffer.contents buf in
  Buffer.reset buf; s


(* print variables such as "A", "B", ..., "Z", "A1", ..., "Z1", "A2", ... *)
let pp_print_var f i =
  pp_print_char f (Char.chr (65 + i mod 26));
  if i > 25 then pp_print_int f (i / 26)

let rec pp_print_term f = function
    V i -> pp_print_var f i
  | A (i, l) ->
      pp_print_string f (try Hashtbl.find no_cnst i with Not_found -> failwith ("pp_print_term: " ^ string_of_int i));
      if l <> [] then pp_enum "," "(" ")" pp_print_term f l

let pp_print_lit f = function
    (i, [l1; l2]) when i = eqn -> pp_print_term f l1; pp_print_string f "="; pp_print_term f l2
  | (i, [l1; l2]) when i = -eqn -> pp_print_term f l1; pp_print_string f "!="; pp_print_term f l2
  | (i, l) ->
      if i < 0 then pp_print_char f '~';
      let s =
        try Hashtbl.find no_cnst (abs i)
        with Not_found -> failwith ("pp_print_lit: " ^ string_of_int i) in
      pp_print_string f s;
      if l <> [] then pp_enum "," "(" ")" pp_print_term f l

let pp_print_clause = pp_enum "," "[" "]" pp_print_lit
let pp_print_clause_vbar = pp_iter "|" pp_print_lit

let pp_print_bin mid fn f (l, r) = fn f l; pp_print_string f mid; fn f r
let pp_print_bin mid fn = pp_enclose "(" ")" (pp_print_bin mid fn)

let rec pp_print_form f = function
    Atom l -> pp_print_lit f l
  | Neg (Atom l) -> pp_print_string f "~"; pp_print_lit f l
  | Neg t -> pp_enclose "~(" ")" pp_print_form f t
  | Conj (l, r) -> pp_print_bin "&" pp_print_form f (l, r)
  | Disj (Neg l, r) -> pp_print_bin "=>" pp_print_form f (l, r)
  | Disj (l, r) -> pp_print_bin "|" pp_print_form f (l, r)
  | Eqiv (l, r) -> pp_print_bin "<=>" pp_print_form f (l, r)
  | Impl (l, r) -> pp_print_bin  "=>" pp_print_form f (l, r)
  | Forall (v, t) -> pp_enclose "![" "]:" pp_print_var f v; pp_print_form f t
  | Exists (v, t) -> pp_enclose "?[" "]:" pp_print_var f v; pp_print_form f t


let rec pp_print_litmat fi f = function
    Lit l -> pp_print_lit f l
  | Mat m -> pp_print_imatrix fi f m
and pp_print_imatrix fi f (i, m) =
  pp_open_box f 1;
  fi f i;
  pp_print_matrix fi f m;
  pp_close_box f ()
and pp_print_matrix fi =
  pp_with_box 1 (pp_enclose_cut "[" "]" (pp_iter_sp "&" (pp_print_iclause fi)))
and pp_print_iclause fi f ((i, v), c) =
  pp_open_box f 1;
  fi f i;
  pp_enum "," "{" "}" pp_print_var f v; pp_print_char f ':'; pp_print_space f ();
  pp_print_nclause fi f c;
  pp_close_box f ()
and pp_print_nclause fi =
  pp_with_box 1 (pp_enclose_cut "[" "]" (pp_iter_sp  "|" (pp_print_litmat fi)))

let pp_print_index = pp_enclose "<" ">" pp_print_int
let pp_print_index_copy = pp_enclose "<" ">" (pp_print_pair "," pp_print_int)

let string_of_lit = print_to_string pp_print_lit
let string_of_form = print_to_string pp_print_form
let string_of_clause = print_to_string pp_print_clause
let string_of_clause_vbar = print_to_string pp_print_clause_vbar
let string_of_iclause c = print_to_string (pp_print_iclause pp_print_null) c
let string_of_iclause_i = print_to_string (pp_print_iclause pp_print_index)
let string_of_iclause_ic = print_to_string (pp_print_iclause pp_print_index_copy)
let string_of_nclause c = print_to_string (pp_print_nclause pp_print_null) c
let string_of_nclause_ic c = print_to_string (pp_print_nclause pp_print_index_copy) c
let string_of_matrix m = print_to_string (pp_print_matrix pp_print_null) m

let string_of_lits = List.map string_of_lit %> String.concat ", "

let print_form = pp_nl pp_print_form std_formatter
let print_clause = pp_print_clause std_formatter
let print_iclause c = pp_nl (pp_print_iclause pp_print_null) std_formatter c
let print_iclause_i = pp_nl (pp_print_iclause pp_print_index) std_formatter
let print_iclause_ic = pp_nl (pp_print_iclause pp_print_index_copy) std_formatter
let print_nclause c = pp_nl (pp_print_nclause pp_print_null) std_formatter c
let print_matrix_i = pp_nl (pp_print_matrix pp_print_index) Format.std_formatter
let print_matrix_ic = pp_nl (pp_print_matrix pp_print_index_copy) Format.std_formatter
