open Batteries
open Term
open Substitution.Substlist

module Clausal =
struct

module Step =
struct

type proof_step = Lem of lit | Pat of lit | Res of (lit * lit list * lit list * int)

let is_res_step = function
    Res _ -> true
  | _ -> false

let show_contr =
  Hashtbl.find Database.Clausal.no_rcont %> uncurry List.cons %> Print.string_of_clause_vbar

let show_proof_step_vbar = function
    Res (lit, path, lem, contr) -> "Ext|" ^ show_contr contr
  | Lem lit -> "Lem|" ^ Print.string_of_lit lit
  | Pat lit -> "Red|" ^ Print.string_of_lit lit

end


module ML =
struct

open Step

let pp_print_res f (lit, path, lem, nlit, npath, nlem, contr) =
  let open Print in
  let print_lits = pp_iter "," pp_print_lit f in
  Format.pp_print_string f (Hashtbl.find Database.Clausal.no_contr contr);
  Format.pp_print_string f ", (";
  pp_print_lit f lit; Format.pp_print_string f "), (";
  print_lits path; Format.pp_print_string f "), (";
  print_lits lem; Format.pp_print_string f "), (";
  pp_print_lit f nlit; Format.pp_print_string f "), (";
  print_lits npath; Format.pp_print_string f "), (";
  print_lits nlem; Format.pp_print_string f ").\n"

let print_res = pp_print_res Format.std_formatter

let res_step sub = function
    Res (lit, path, lem, contr) ->
      let inst = inst_lit sub in
      Some (inst lit, List.map inst path, List.map inst lem, lit, path, lem, contr)
  | _ -> None

let print_proof sub =
  List.filter_map (res_step sub) %> List.rev %> List.iter print_res

end


module Proof =
struct

type ('l, 'c) proof =
    Lemma
  | Reduction
  | Extension of 'c * ('l * ('l, 'c) proof) list

open Print
open Format

let rec pp_print_proof (fl, fc) sub f (lit, prf) =
  pp_open_vbox f 2;
  pp_print_string f (fl sub lit); pp_print_char f ' ';
  pp_print_prf (fl, fc) sub f prf;
  pp_close_box f ()
and pp_print_prf (fl, fc) sub f = function
    Reduction -> pp_print_string f "Red"
  | Lemma -> pp_print_string f "Lem"
  | Extension (cl, prfs) ->
      pp_print_string f ("Ext " ^ fc cl);
      pp_print_proofs (fl, fc) sub f prfs
and pp_print_proofs (fl, fc) sub f =
  List.iter (fun x -> pp_print_cut f (); pp_print_proof (fl, fc) sub f x)

let print_proof (fl, fc) sub prf = pp_nl (pp_print_proof (fl, fc) sub) std_formatter prf

let string_of_sub_lit sub = string_of_lit % inst_lit sub

let print_proof_hsh prf =
  print_proof (string_of_sub_lit, Hashtbl.find Database.Clausal.no_contr) prf

let rec of_steps = function
  Step.Lem lit :: tl -> tl, (lit, Lemma)
| Step.Pat lit :: tl -> tl, (lit, Reduction)
| Step.Res (lit, _, _, hsh) :: tl ->
    let (_, cla) = Hashtbl.find Database.Clausal.no_rcont hsh in
    let tl', prfs = fold_right_map (fun acc _ -> of_steps acc) tl cla in
    tl', (lit, Extension (hsh, List.rev prfs))
| _ -> failwith "of_steps"

let of_sub_steps (sub, prf) = (sub, prf |> List.rev |> of_steps |> snd |> snd)

let rec to_steps lit pat lem = function
    Lemma -> [Step.Lem lit]
  | Reduction -> [Step.Pat lit]
  | Extension (cl, prfs) ->
      let res = Step.Res (lit, pat, lem, cl) in
      let pat' = lit :: pat in
      let go lem' (l, prf) = l :: lem', to_steps l pat' lem' prf in
      res :: List.concat (fold_map go lem prfs |> snd)

end

end


module Nonclausal =
struct

type ('l, 'c, 're) proof =
    Lemma
  | Reduction
  | Decomposition of 'c *       ('l * ('l, 'c, 're) proof) list
  | Extension     of 'c * 're * ('l * ('l, 'c, 're) proof) list

open Print
open Format

let rec pp_print_proof (fl, fc) sub f (lit, prf) =
  pp_open_vbox f 2;
  pp_print_string f (fl sub lit); pp_print_char f ' ';
  pp_print_prf (fl, fc) sub f prf;
  pp_close_box f ()
and pp_print_prf (fl, fc) sub f = function
    Reduction -> pp_print_string f "Red"
  | Lemma -> pp_print_string f "Lem"
  | Decomposition (cl, prfs) ->
      pp_print_string f ("Dec " ^ fc cl);
      pp_print_proofs (fl, fc) sub f prfs
  | Extension (cl, recon, prfs) ->
      pp_print_string f ("Ext " ^ fc cl);
      pp_print_proofs (fl, fc) sub f prfs
and pp_print_proofs (fl, fc) sub f =
  List.iter (fun x -> pp_print_cut f (); pp_print_proof (fl, fc) sub f x)

let print_proof (fl, fc) sub prf = pp_nl (pp_print_proof (fl, fc) sub) std_formatter prf

let string_of_sub_lit sub lit = string_of_lit (inst_lit sub lit)
let string_of_sub_litmat sub =
  Matrix.Nonclausal.map_litmat (string_of_sub_lit sub) (snd %> string_of_matrix)

let print_proof_ncl prf =
  print_proof (string_of_sub_litmat, Print.string_of_nclause_ic) prf

end
