open Term

module type Substitution = sig
  type t
  exception Unify

  val unify_lit : t -> lit -> lit -> t
  val unify_tms : t -> iterm list -> iterm list -> t
  val unify_tms_off : int -> t -> iterm list -> iterm list -> t
  val eq_lit : t -> lit -> lit -> bool
  val inst_lit : t -> lit -> lit
  val ground_lit : t -> lit -> bool
  val empty : int -> t
  val to_list : t -> (int * iterm) list
end

module Substlist : Substitution with type t = (int * iterm) list =
struct

exception Unify

type t = (int * iterm) list

let rec istriv sub x = function
    V y -> (*Printf.printf "V: %d %d\n" x y;*) y = x || (try let t = List.assoc y sub in istriv sub x t with Not_found -> false)
  | A (f, a) -> List.exists (istriv sub x) a && raise Unify;;

let add_subst sub x t =
  if istriv sub x t then sub else (x, t) :: sub

let rec unify_tm sub tm1 tm2 = match tm1,tm2 with
    A (f,fargs), A (g,gargs) -> if f <> g then raise Unify
      else unify_tms sub fargs gargs
  | tm, V x | V x, tm ->
      (try let t = List.assoc x sub in unify_tm sub tm t
      with Not_found -> add_subst sub x tm)
and unify_tms sub l1 l2 = List.fold_left2 unify_tm sub l1 l2

let unify_lit env (h1, l1) (h2, l2) =
  if h1 <> h2 then raise Unify else unify_tms env l1 l2

(* Unification with renaming of the second argument *)
let rec unify_tm_off off sub t1 t2 = match t1,t2 with
    A (f,fargs), A (g,gargs) -> if f <> g then raise Unify else
        unify_tms_off off sub fargs gargs
  | _, V x -> let x = x + off in
     (try let t = List.assoc x sub in unify_tm_off 0 sub t1 t
     with Not_found -> add_subst sub x t1)
  | V x, _ ->
     (try let t = List.assoc x sub in unify_tm_off off sub t t2
     with Not_found -> add_subst sub x (offset_term off t2))
and unify_tms_off off = List.fold_left2 (unify_tm_off off)

let rec eq_var_tm sub x = function
    V y -> y = x || (try let t = List.assoc y sub in eq_var_tm sub x t with Not_found -> false)
  | A (f, a) -> false

let rec eq_term sub tm1 tm2 =
  match tm1,tm2 with
    A (f,fa), A (g,ga) -> f = g && eq_terms sub fa ga
  | _, V x -> (try let t = List.assoc x sub in eq_term sub tm1 t with Not_found -> eq_var_tm sub x tm1)
  | V x, _ -> (try let t = List.assoc x sub in eq_term sub t tm2 with Not_found -> eq_var_tm sub x tm2)
and eq_terms sub = List.for_all2 (eq_term sub)

let eq_lit sub (p,pa) (q,qa) = p = q && eq_terms sub pa qa

(* In leanCoP: only for printing, in resolve is used by unify_rename2 *)
let rec inst_tm sub = function
    V v -> (try let t = List.assoc v sub in inst_tm sub t with Not_found -> V v)
  | A (f,args) -> A (f, List.map (inst_tm sub) args)
let inst_lit sub (p, l) = (p, List.map (inst_tm sub) l)

let rec ground_tm sub = function
    V v -> (try let t = List.assoc v sub in ground_tm sub t with Not_found -> false)
  | A (f,args) -> List.for_all (ground_tm sub) args
let ground_lit sub (p, l) = List.for_all (ground_tm sub) l

let empty _ = []

let to_list sub = sub

end


module Substarray : Substitution =
struct

type t = (iterm option array * int list ref) * int list

exception Unify

let rec restore_subst ((suba, env), subl) =
  while not (!env == subl) do
    match !env with
      h::t -> suba.(h) <- None; env := t
    | [] -> failwith "restore_subst"
  done

let rec istriv suba x = function
  | V y -> y = x || (match suba.(y) with Some t -> istriv suba x t | None -> false)
  | A (f, a) -> List.exists (istriv suba x) a && raise Unify

let add_subst ((suba, env), subl as sub) x tm =
  if istriv suba x tm then sub
  else (suba.(x) <- Some tm; env := x :: !env; ((suba, env), !env))

let rec unify_tm ((suba, env), subl as sub) tm1 tm2 = match tm1,tm2 with
  | A (f,fargs), A (g,gargs) -> if f <> g then raise Unify
      else List.fold_left2 unify_tm sub fargs gargs
  | tm, V x | V x, tm -> (match suba.(x) with
    | Some t -> unify_tm sub tm t
    | None -> add_subst sub x tm)

let unify_tms sub l1 l2 = restore_subst sub; List.fold_left2 unify_tm sub l1 l2

let unify_lit env ((h1 : int), l1) (h2, l2) =
  if h1 <> h2 then raise Unify else unify_tms env l1 l2

let rec offset_vars off = function
  | V x -> V (x + off)
  | A (x, l) -> A (x, List.map (offset_vars off) l)

(* Unification with renaming of the second argument *)
let rec unify_tm_off off ((suba, env), subl as sub) t1 t2 = match t1,t2 with
  | A (f,fargs), A (g,gargs) -> if f <> g then raise Unify else
        List.fold_left2 (unify_tm_off off) sub fargs gargs
  | _, V x -> let x = x + off in (match suba.(x) with
     | Some t -> unify_tm_off 0 sub t1 t
     | None -> add_subst sub x t1)
  | V x,_ -> (match suba.(x) with
     | Some t -> unify_tm_off off sub t t2
     | None -> let t2' = offset_vars off t2 in add_subst sub x t2')

let unify_tms_off off sub l1 l2 =
  restore_subst sub; List.fold_left2 (unify_tm_off off) sub l1 l2


let rec eq_var_tm suba x = function
  | V y -> y = x || (match suba.(y) with Some t -> eq_var_tm suba x t | None -> false)
  | A (f, a) -> false

let rec eq_tm suba tm1 tm2 =
  match tm1,tm2 with
  | A (f,fargs), A (g,gargs) -> f = g && List.for_all2 (eq_tm suba) fargs gargs
  | _, V x -> (match suba.(x) with Some t -> eq_tm suba tm1 t | None -> eq_var_tm suba x tm1)
  | V x, _ -> (match suba.(x) with Some t -> eq_tm suba t tm2 | None -> eq_var_tm suba x tm2)

let eq_lit ((suba, _), _ as sub) (p1,args1) (p2,args2) =
  p1 = p2 && (restore_subst sub; List.for_all2 (eq_tm suba) args1 args2)

let empty n =
  let suba = Array.make n (None : iterm option)
  and env = ref []
  and subl = []
  in ((suba, env), subl)

let to_list ((suba, env), subl) = List.map
  (fun v -> match suba.(v) with None -> failwith "convert_subst" | Some t -> v, t) subl

let rec inst_term suba = function
    V x -> (match suba.(x) with Some t -> inst_term suba t | None -> V x)
  | A (f, a) -> A (f, List.map (inst_term suba) a)

let inst_lit ((suba, _), _ as sub) (p, l) =
  restore_subst sub; (p, List.map (inst_term suba) l)

let rec ground_tm suba = function
    V x -> (match suba.(x) with Some t -> ground_tm suba t | None -> false)
  | A (f, a) -> List.for_all (ground_tm suba) a

let ground_lit ((suba, _), _ as sub) (p, l) =
  restore_subst sub; List.for_all (ground_tm suba) l

end

module Substoff (Sub : Substitution) = struct

include Sub

let eq (sub, off) l1 l2 = eq_lit sub l1 l2

let unify (sub, off) l1 l2 = try Some (unify_lit sub l1 l2, off) with Unify -> None

let unify_rename_subst off l1 l2 sub list =
  unify_tms_off off sub l1 l2, List.map (offset_lit off) list

let unify_rename (s, off) args1 (args2, rest, vars, _) =
  try Some (if vars = 0 then ((unify_tms s args1 args2, off), rest) else
    let s, rest = unify_rename_subst off args1 args2 s rest in ((s, off + vars), rest))
  with Unify -> None
  
end
