open Batteries

module IM = Map.Make(Int)
module IS = Set.Make(Int)

type 'v term = V of 'v
             | A of int * 'v term list
type iterm = int term

(* For efficiency in CNF: int * term list, with negative for negation *)
type lit = int * int term list

let negate_lit (i, l) = (-i, l)

let rec map_term_vars f = function
    V v -> f v
  | A (p, t) -> A (p, List.map (map_term_vars f) t)
let map_lit_vars f (p, a) = (p, List.map (map_term_vars f) a)

let map_term fv fa = function
    V v -> fv v
  | A (p, a) -> fa (p, a)

(* set all variable indices to zero *)
let normalize t = map_term_vars (const (V 0)) t

let rec offset_term off tm = match tm with
    V v -> V (v + off)
  | A (f, a) -> A (f, List.map (offset_term off) a)

let offset_lit off (p, pa) = p, List.map (offset_term off) pa

let fold_left_map f acc = List.fold_left
  (fun (acc, ys) x -> let (acc', y) = f acc x in (acc', y :: ys)) (acc, [])

let fold_right_map f acc xs = List.fold_right
  (fun x (acc, ys) -> let (acc', y) = f acc x in (acc', y :: ys)) xs (acc, [])

let fold_map f sf l = let (sf, rev) = fold_left_map f sf l in sf, List.rev rev

let rec rename_term ((map, next) as sf) = function
    V i -> (try (sf, V (IM.find i map)) with Not_found -> (IM.add i next map, next + 1), V next)
  | A (i, l) -> let sf, l = fold_map rename_term sf l in sf, A (i, l)

let rename_lit sf (i, l) = let sf, l = fold_map rename_term sf l in sf, (i, l)


let rec term_ground t = map_term (const false) lit_ground t
and lit_ground (p, a) = List.for_all term_ground a

let rec term_max_var acc = function
    V x -> max acc x
  | A (_, l) -> List.fold_left term_max_var acc l
let lit_max_var acc (_, a) = List.fold_left term_max_var acc a

let rec term_vars sf = function
    V x -> IS.add x sf
  | A (_, l) -> List.fold_left term_vars sf l
let lit_vars sf (p, a) = List.fold_left term_vars sf a

let rec term_funs f = function
    A (i, t) -> List.fold_left term_funs (if t <> [] then IM.add i (List.length t) f else f) t
  | _ -> f
